import {HttpClient} from "@angular/common/http";
import {environment} from "@environments/environment";
import {TranslateLoader} from "@ngx-translate/core";
import {
  combineLatest,
  Observable,
} from "rxjs";
import {map} from "rxjs/operators";

export class TranslateHttpLoaderExtra implements TranslateLoader {
  private readonly prefix: string = "./assets/";
  private readonly prefixBase: string = "i18n/";
  private readonly prefixThemes: string = "themes/";
  private readonly prefixCustomizable: string = "i18n/";

  private readonly customizableSuffix: string = "_customizable";

  constructor(private http: HttpClient, public suffix: string = ".json") {
  }

  /**
   * Gets the translations from the server
   */
  public getTranslation(lang: string): Observable<Object> {
    return combineLatest([
      this.http.get(`${this.prefix}${this.prefixBase}${lang}${this.suffix}`),
      this.http.get(`${this.prefix}${this.prefixThemes}${environment.theme}/i18n/${lang}_${environment.theme}${this.suffix}`),
      this.http.get(`${this.prefix}${this.prefixCustomizable}${lang}${this.customizableSuffix}${this.suffix}`),
    ]).pipe(
      map(([base, theme, extra]) => mergeDeep(base, theme, extra)),
    );
  }
}

/**
 * Simple object check.
 * @param item
 * @returns {boolean}
 */
function isObject(item: any): boolean {
  return (item && typeof item === "object" && !Array.isArray(item));
}

/**
 * Deep merge two objects.
 * @param target
 * @param ...sources
 */
function mergeDeep(target: Object, ...sources: any): Object {
  if (!sources.length) {
    return target;
  }
  const source = sources.shift();

  if (isObject(target) && isObject(source)) {
    for (const key in source) {
      if (isObject(source[key])) {
        if (!target[key]) {
          Object.assign(target, {[key]: {}});
        }
        mergeDeep(target[key], source[key]);
      } else {
        Object.assign(target, {[key]: source[key]});
      }
    }
  }

  return mergeDeep(target, ...sources);
}
