import {Params} from "@angular/router";
import {environment} from "@environments/environment";
import {
  isEmptyString,
  isNotNullNorUndefined,
  isNullOrUndefined,
  MappingObject,
  MappingObjectUtil,
  StringUtil,
} from "solidify-frontend";

export class UrlQueryParamHelper {
  static readonly QUERY_PARAM_OAUTH_CODE: string = "code";
  static readonly QUERY_PARAM_OAUTH_STATE: string = "state";
  private static readonly _QUERY_PARAM_ORCID: string = environment.orcidQueryParam;
  private static readonly _QUERY_PARAM_START_CHAR: string = "?";
  private static readonly _QUERY_PARAM_SEPARATOR_CHAR: string = "&";
  private static readonly _QUERY_PARAM_VALUES_SEPARATOR: string = environment.nonceStateSeparator;
  private static readonly _QUERY_PARAM_VALUE_AFFECTION_CHAR: string = "=";

  static getQueryParamValueSeparator(): string {
    return this._QUERY_PARAM_VALUE_AFFECTION_CHAR;
  }

  static getQueryParamMappingObject(): MappingObject<string | undefined> {
    return this.getQueryParamMappingObjectFromRawQueryParamString(this._getQueryParameters());
  }

  static getQueryParamMappingObjectFromRawQueryParamString(rawQueryParam: string): MappingObject<string | undefined> {
    const mappingObject = new Map<string, string>();
    if (isNullOrUndefined(rawQueryParam)) {
      return mappingObject;
    }
    const listQueryParam = this._getListQueryParameters(rawQueryParam);
    if (listQueryParam.length === 0) {
      return mappingObject;
    }
    listQueryParam.forEach(queryParam => {
      const queryParamSplitted = queryParam.split(this._QUERY_PARAM_VALUE_AFFECTION_CHAR);
      mappingObject.set(queryParamSplitted[0], queryParamSplitted.length > 1 ? queryParamSplitted[1] : undefined);
    });

    return mappingObject;
  }

  static getQueryParamOAuth2(): string {
    // NEED TO PRESERVE QUERY PARAM WHEN ROOTING USING HASH STRATEGY IS INITIALIZED
    // NECESSARY FOR OAUTH 2 PROCESS (SHOULD PRESERVED CODE AND STATE IN QUERY PARAM)
    const queryParamString = this._getQueryParameters();
    if (isNullOrUndefined(queryParamString)) {
      return StringUtil.stringEmpty;
    }
    let listQueryParam = this._getListQueryParameters(queryParamString);
    listQueryParam = this._preserveOnlyQueryParametersOauth(listQueryParam);

    if (listQueryParam.length === 0) {
      return StringUtil.stringEmpty;
    }
    return this._getQueryParametersString(listQueryParam);
  }

  static getStateFromUrlWithQueryParamOAuth2(url: string): string | undefined {
    const splitted = url.split(this._QUERY_PARAM_START_CHAR);
    const queryParamString = splitted.length > 1 ? splitted[1] : undefined;
    if (isNullOrUndefined(queryParamString)) {
      return StringUtil.stringEmpty;
    }

    const listQueryParam = this._getListQueryParameters(queryParamString);
    const stateQueryParam: string = this._getStateFromQueryParameter(listQueryParam);

    if (isNullOrUndefined(stateQueryParam)) {
      return StringUtil.stringEmpty;
    }
    return UrlQueryParamHelper._getStateValuesFromQueryParam(stateQueryParam);
  }

  private static _getStateValuesFromQueryParam(stateQueryParam: string): string {
    //the state is create as : nonce + nonceStateSeparator + state
    const stateValues = stateQueryParam.replace(this.QUERY_PARAM_OAUTH_STATE + this._QUERY_PARAM_VALUE_AFFECTION_CHAR, "");
    const splitted = stateValues.split(this._QUERY_PARAM_VALUES_SEPARATOR);
    const stateValue = splitted.length > 1 ? splitted[1] : undefined;
    return stateValue === undefined ? stateValue : decodeURIComponent(stateValue);
  }

  static getQueryParameterWithoutUnwantedQueryParam(): Params | undefined {
    const queryParamString = this._getQueryParameters();
    const params = {};
    if (isNullOrUndefined(queryParamString)) {
      return params;
    }
    let listQueryParam = this._getListQueryParameters(queryParamString);
    listQueryParam = this._removeOnlyUnwantedQueryParameters(listQueryParam);

    if (listQueryParam.length === 0) {
      return params;
    }
    listQueryParam.forEach(queryParam => {
      const queryParamSpitted = queryParam.split(this._QUERY_PARAM_VALUE_AFFECTION_CHAR);
      MappingObjectUtil.set(params, queryParamSpitted[0], queryParamSpitted.length > 1 ? queryParamSpitted[1] : StringUtil.stringEmpty);
    });
    return params;
  }

  static getListQueryParams(url: string): string[] | undefined {
    if (isNullOrUndefined(url)) {
      return undefined;
    }
    const listQueryParam = this._getListQueryParameters(url);
    return this._getListQueryParams(listQueryParam);
  }

  static contains(desiredQueryParam: string): boolean {
    if (isNullOrUndefined(desiredQueryParam) || isEmptyString(desiredQueryParam)) {
      return false;
    }
    const queryParamString = this._getQueryParameters();
    if (isNullOrUndefined(queryParamString)) {
      return false;
    }
    const listQueryParam = this._getListQueryParameters(queryParamString);
    return isNotNullNorUndefined(listQueryParam.find(queryParam => queryParam.startsWith(desiredQueryParam + this._QUERY_PARAM_VALUE_AFFECTION_CHAR)));
  }

  private static _preserveOnlyQueryParametersOauth(listQueryParam: string[]): string[] {
    return listQueryParam.filter(query => query.startsWith(this.QUERY_PARAM_OAUTH_CODE + this._QUERY_PARAM_VALUE_AFFECTION_CHAR)
      || query.startsWith(this.QUERY_PARAM_OAUTH_STATE + this._QUERY_PARAM_VALUE_AFFECTION_CHAR));
  }

  private static _removeOnlyUnwantedQueryParameters(listQueryParam: string[]): string[] {
    return listQueryParam.filter(query => !query.startsWith(this.QUERY_PARAM_OAUTH_CODE + this._QUERY_PARAM_VALUE_AFFECTION_CHAR)
      && !query.startsWith(this.QUERY_PARAM_OAUTH_STATE + this._QUERY_PARAM_VALUE_AFFECTION_CHAR)
      && !query.startsWith(this._QUERY_PARAM_ORCID + this._QUERY_PARAM_VALUE_AFFECTION_CHAR),
    );
  }

  private static _getQueryParameters(): string | undefined {
    const splitted = window.location.href.split(this._QUERY_PARAM_START_CHAR);
    return splitted.length > 1 ? splitted[1] : undefined;
  }

  private static _getListQueryParameters(queryParam: string): string[] {
    return queryParam.split(this._QUERY_PARAM_SEPARATOR_CHAR);
  }

  private static _getQueryParametersString(listQueryParam: string[]): string {
    return this._QUERY_PARAM_START_CHAR + listQueryParam.join(this._QUERY_PARAM_SEPARATOR_CHAR);
  }

  private static _getStateFromQueryParameter(listQueryParam: string[]): string {
    return listQueryParam.find(query => query.startsWith(this.QUERY_PARAM_OAUTH_STATE + this._QUERY_PARAM_VALUE_AFFECTION_CHAR));
  }

  private static _getListQueryParams(listQueryParam: string[]): string[] {
    const stateQueryParam: string = this._getStateFromQueryParameter(listQueryParam);
    //the state is create as : nonce + nonceStateSeparator + state
    const stateValues = stateQueryParam.replace(this.QUERY_PARAM_OAUTH_STATE + this._QUERY_PARAM_VALUE_AFFECTION_CHAR, "");
    const splitted = stateValues.split(this._QUERY_PARAM_VALUES_SEPARATOR);
    // Get query params. Url form as: code;url-to-redirect;queryParameters
    const queryParams = splitted.length > 2 ? decodeURIComponent(splitted[2]) : undefined;
    if (queryParams !== undefined) {
      return queryParams.split(this._QUERY_PARAM_SEPARATOR_CHAR);
    }
    return undefined;
  }

}

