import {
  isNonEmptyString,
  isNotNullNorUndefined,
  isNullOrUndefined,
  isNumberReal,
} from "solidify-frontend";

export class StorageHelper<StorageKeyEnum> {
  protected static readonly storage: Storage;
  private static readonly LIST_SEPARATOR: string = ";";

  static getItem<StorageKeyEnum>(key: StorageKeyEnum): string {
    return this.storage.getItem(key.toString());
  }

  static setItem<StorageKeyEnum>(key: StorageKeyEnum, value: string): void {
    this.storage.setItem(key.toString(), value);
  }

  static removeItem<StorageKeyEnum>(key: StorageKeyEnum): void {
    this.storage.removeItem(key.toString());
  }

  static getListItem<StorageKeyEnum>(key: StorageKeyEnum): string[] {
    let list = [];
    const listSerialized = this.storage.getItem(key.toString());
    if (isNotNullNorUndefined(listSerialized) && isNonEmptyString(listSerialized)) {
      list = listSerialized.split(this.LIST_SEPARATOR);
    }
    return list;
  }

  static initItemInList<StorageKeyEnum>(key: StorageKeyEnum, list: string[]): string[] {
    if (isNullOrUndefined(list)) {
      list = [];
    }
    this.storage.setItem(key.toString(), list.join(this.LIST_SEPARATOR));
    return list;
  }

  static addItemInList<StorageKeyEnum>(key: StorageKeyEnum, item: string, maxItems: number | undefined = undefined): string[] {
    let list = [];
    const listSerialized = this.storage.getItem(key.toString());
    if (isNonEmptyString(listSerialized)) {
      list = listSerialized.split(this.LIST_SEPARATOR);
    }
    if (!list.includes(item)) {
      list.push(item);
      if (isNumberReal(maxItems) && list.length > maxItems) {
        list = list.splice(list.length - maxItems, list.length);
      }
      this.storage.setItem(key.toString(), list.join(this.LIST_SEPARATOR));
    }

    return list;
  }

  static removeItemInList<StorageKeyEnum>(key: StorageKeyEnum, item: string): string[] {
    let list = [];
    const listSerialized = this.storage.getItem(key.toString());
    if (!isNullOrUndefined(listSerialized)) {
      list = listSerialized.split(this.LIST_SEPARATOR);
    }
    const indexOfItem = list.indexOf(item);
    if (indexOfItem !== -1) {
      list.splice(indexOfItem, 1);
      if (list.length === 0) {
        this.storage.removeItem(key.toString());
      } else {
        this.storage.setItem(key.toString(), list.join(this.LIST_SEPARATOR));
      }
    }
    return list;
  }
}
