/* tslint:disable:no-console */
import {environment} from "@environments/environment";
import {intervalBackoff} from "backoff-rxjs";
import {IntervalBackoffConfig} from "backoff-rxjs/observable/intervalBackoff";
import {exponentialBackoffDelay} from "backoff-rxjs/utils";
import {
  fromEvent,
  Observable,
} from "rxjs";
import {
  filter,
  map,
  sampleTime,
  startWith,
  switchMap,
  takeWhile,
  tap,
} from "rxjs/operators";
import {
  isFalse,
  isNumberFinite,
  isTrue,
  isUndefined,
} from "solidify-frontend";

export class PollingHelper<Type> {
  private static SECOND_TO_MILLISECOND: number = 1000;
  private static LOG_INFOS: boolean = false;

  static startPollingObs<Type>(config: PollingConfigInfo<Type>): Observable<Type> {
    const stopAfterMaxIntervalReached = isTrue(config.incrementInterval) && isTrue(config.stopRefreshAfterMaximumIntervalReached);
    let maxIntervalReached = false;
    const maxIntervalInSecond = isNumberFinite(config.maximumIntervalRefreshInSecond) ? config.maximumIntervalRefreshInSecond : environment.pollingMaxIntervalInSecond;

    const intervalInMillisecond = config.initialIntervalRefreshInSecond * this.SECOND_TO_MILLISECOND;
    const optionIntervalBackoff = {
      initialInterval: intervalInMillisecond,
      maxInterval: maxIntervalInSecond * this.SECOND_TO_MILLISECOND,
      backoffDelay: (iteration, initialInterval) => {
        if (isTrue(config.incrementInterval)) {
          const interval = exponentialBackoffDelay(iteration, initialInterval);
          if (isTrue(stopAfterMaxIntervalReached) && isFalse(maxIntervalReached)) {
            maxIntervalReached = interval >= maxIntervalInSecond * this.SECOND_TO_MILLISECOND;
            if (this.LOG_INFOS && maxIntervalReached) {
              console.info("Max interval rechead");
            }
          }
          return interval;
        } else {
          return initialInterval;
        }
      },
    } as IntervalBackoffConfig;

    const observable = intervalBackoff(optionIntervalBackoff).pipe(
      filter(() => {
        const shouldContinue = isFalse(stopAfterMaxIntervalReached) || isFalse(maxIntervalReached);
        if (this.LOG_INFOS && isTrue(stopAfterMaxIntervalReached) && isFalse(shouldContinue)) {
          console.info("Stop action (Max interval rechead)");
        }
        return shouldContinue;
      }),
      takeWhile(() => isUndefined(config.continueUntil) || config.continueUntil()),
      tap(() => isUndefined(config.doBeforeFilter) || config.doBeforeFilter()),
      filter(() => isUndefined(config.filter) || config.filter()),
      map(() => config.actionToDo()),
    );

    if (isTrue(config.incrementInterval) && isTrue(config.resetIntervalWhenUserMouseEvent)) {
      return fromEvent(document, "mousemove").pipe(
        sampleTime(intervalInMillisecond),
        tap(() => {
          if (this.LOG_INFOS) {
            console.info("Reset interval due to user mouse event");
          }
          maxIntervalReached = false;
        }),
        startWith(null),
        switchMap(() => observable),
      );
    } else {
      return observable;
    }
  }
}

export interface PollingConfigInfo<Type> {
  initialIntervalRefreshInSecond: number;
  maximumIntervalRefreshInSecond?: number | undefined;
  doBeforeFilter?: () => void | undefined;
  continueUntil?: () => boolean | undefined;
  filter?: () => boolean | undefined;
  actionToDo: () => Type;
  incrementInterval?: boolean | undefined;
  stopRefreshAfterMaximumIntervalReached?: boolean | undefined;
  resetIntervalWhenUserMouseEvent?: boolean | undefined;
}
