import {
  Actions,
  ofActionCompleted,
  Store,
} from "@ngxs/store";
import {FormErrorEnum} from "@shared/enums/form-error.enum";
import {FormControlKey} from "@shared/models/form-control-key.model";
import {Observable} from "rxjs";
import {
  map,
  take,
} from "rxjs/operators";
import {
  BaseResource,
  isNullOrUndefined,
  MappingObjectUtil,
  QueryParameters,
  ResourceActionHelper,
  ResourceNameSpace,
  StringUtil,
} from "solidify-frontend";

export class CrudHelper {

  private static checkSingleConstraintAvailable(formControlKey: FormControlKey, store: Store, actions$: Actions, resourceNameSpace: ResourceNameSpace, resId: string | undefined): Observable<boolean> {
    const queryParameters = new QueryParameters();
    MappingObjectUtil.set(queryParameters.search.searchItems, formControlKey.key, formControlKey.formControl.value);
    store.dispatch(ResourceActionHelper.getAll(resourceNameSpace, queryParameters));
    return actions$.pipe(ofActionCompleted(resourceNameSpace.GetAllSuccess))
      .pipe(
        take(1),
        map(result => {
          if (result.result.successful && result.action.list._data.length > 0) {
            if (result.action.list._data.filter((f: BaseResource) =>
              (StringUtil.stringEmpty + f[formControlKey.key]).toLowerCase() === (StringUtil.stringEmpty + formControlKey.formControl.value).toLowerCase()
              && (isNullOrUndefined(resId) || f.resId !== resId)).length === 0) {
              return true;
            }
            formControlKey.formControl.setErrors({
              errorsFromBackend: [FormErrorEnum.ALREADY_USED],
            });
            formControlKey.formControl.markAsTouched();
            formControlKey.changeDetector.detectChanges();
            return false;
          }
        }),
      );
  }

  private static checkMultipleConstraintAvailable(formControlKey: FormControlKey, store: Store, actions$: Actions, resourceNameSpace: ResourceNameSpace, resId: string | undefined): Observable<boolean> {
    const queryParameters = new QueryParameters();
    const keys = formControlKey.keys;
    keys.forEach(k => {
      MappingObjectUtil.set(queryParameters.search.searchItems, k, formControlKey.form.get(k).value);
    });
    store.dispatch(ResourceActionHelper.getAll(resourceNameSpace, queryParameters));
    return actions$.pipe(ofActionCompleted(resourceNameSpace.GetAllSuccess))
      .pipe(
        take(1),
        map(result => {
          if (result.result.successful && result.action.list._data.length > 0) {
            let list = result.action.list._data.filter((f: BaseResource) => isNullOrUndefined(resId) || f.resId !== resId);
            keys.forEach(k => {
              const value = formControlKey.form.get(k).value;
              list = list.filter(t => (StringUtil.stringEmpty + t[k]).toLowerCase() === (StringUtil.stringEmpty + value).toLowerCase());
            });
            if (list.length === 0) {
              return true;
            }

            keys.forEach(k => {
              const formControl = formControlKey.form.get(k);
              formControl.setErrors({
                errorsFromBackend: [FormErrorEnum.ALREADY_USED],
              });
            });
            formControlKey.form.markAsTouched();
            formControlKey.changeDetector.detectChanges();
            return false;
          } else {
            keys.forEach(k => {
              const formControl = formControlKey.form.get(k);
              formControl.updateValueAndValidity();
            });
            formControlKey.form.markAsTouched();
            formControlKey.changeDetector.detectChanges();
          }
        }),
      );
  }

  static checkAvailable(formControlKey: FormControlKey, store: Store, actions$: Actions, resourceNameSpace: ResourceNameSpace, resId: string | undefined): Observable<boolean> {
    if (!isNullOrUndefined(formControlKey.keys) && !isNullOrUndefined(formControlKey.form)) {
      return this.checkMultipleConstraintAvailable(formControlKey, store, actions$, resourceNameSpace, resId);
    } else if (!isNullOrUndefined(formControlKey.key) && !isNullOrUndefined(formControlKey.formControl)) {
      return this.checkSingleConstraintAvailable(formControlKey, store, actions$, resourceNameSpace, resId);
    }
    throw new Error("Should provide key and formControl for single constraint or key and form for multiple constraint");
  }
}
