import {Checksum} from "@deposit/models/checksum.model";
import {Enums} from "@enums";
import {ArchiveDataFile} from "@home/models/archive-data-file.model";
import {
  ArchiveMetadataDataFile,
  MetadataDataFileEnum,
} from "@shared/models/business/archive-metadata-data-file.model";
import {FileSizePipe} from "@shared/pipes/file-size.pipe";
import {
  isNullOrUndefined,
  isTruthyObject,
  MappingObjectUtil,
} from "solidify-frontend";

export class ArchiveDataFileHelper {
  private static _fileSizePipe: FileSizePipe = new FileSizePipe();

  static adaptListArchivesMetadataInArchive(listArchivesMetadata: ArchiveMetadataDataFile[]): ArchiveDataFile[] {
    const newList: ArchiveDataFile[] = [];

    listArchivesMetadata.forEach((archive: ArchiveMetadataDataFile) => {
      newList.push(this.adaptArchiveMetadataInArchiveDataFile(archive));
    });
    return newList;
  }

  static adaptArchiveMetadataInArchiveDataFile(archiveMetadata: ArchiveMetadataDataFile): ArchiveDataFile {
    if (isNullOrUndefined(archiveMetadata)) {
      return undefined;
    }
    if (archiveMetadata.metadata.type === "aip") {
      return this._adaptArchiveMetadataInArchiveDataFileCollection(archiveMetadata);
    } else {
      return this._adaptArchiveMetadataInArchiveDataFileFile(archiveMetadata);
    }
  }

  private static _adaptArchiveMetadataInArchiveDataFileFile(archiveMetadata: ArchiveMetadataDataFile): ArchiveDataFile {
    const archiveDataFile = {
      resId: archiveMetadata.resId,
      creation: {
        when: archiveMetadata.metadata.events?.creation?.date,
      } as any,
      lastUpdate: null,
      sourceData: null,
      status: null,
      statusMessage: null,
      virusCheck: {
        checkDate: archiveMetadata.metadata.events?.[MetadataDataFileEnum.eventsVirusCheck]?.date,
      } as any,
      smartSize: this._fileSizePipe.transform(archiveMetadata.metadata?.file?.size),
      relativeLocation: archiveMetadata.metadata?.file?.path,
      packageId: archiveMetadata.metadata?.archiveId,
      initialPath: null,
      finalData: null,
      fileSize: archiveMetadata.metadata?.file?.size,
      fileName: archiveMetadata.metadata?.file?.name,
      fileFormat: {
        format: archiveMetadata.metadata?.format?.description,
        md5: archiveMetadata.metadata?.checksums?.MD5,
        puid: archiveMetadata.metadata?.format?.PRONOM,
        version: archiveMetadata.metadata?.format?.version,
      },
      dataCategory: archiveMetadata.metadata?.dlcm?.category,
      dataType: archiveMetadata.metadata?.dlcm?.type,
      metadataType: null,
      complianceLevel: Enums.DataFile.ConvertComplianceLevel.convertNumberToString(archiveMetadata.metadata?.dlcm?.complianceLevel),
      checksums: this._getCheckums(archiveMetadata),
      available: null,
    } as ArchiveDataFile;
    if (MappingObjectUtil.size(archiveDataFile.virusCheck as any) === 0) {
      archiveDataFile.virusCheck = undefined;
    }
    return archiveDataFile;
  }

  private static _adaptArchiveMetadataInArchiveDataFileCollection(archiveMetadata: ArchiveMetadataDataFile): ArchiveDataFile {
    return {
      resId: archiveMetadata.metadata?.aip?.id,
      lastUpdate: null,
      sourceData: null,
      status: null,
      statusMessage: null,
      smartSize: this._fileSizePipe.transform(archiveMetadata.metadata?.aip?.size),
      relativeLocation: null,
      packageId: archiveMetadata.metadata?.archiveId,
      initialPath: null,
      finalData: null,
      fileSize: archiveMetadata.metadata?.aip?.size,
      fileName: archiveMetadata.resId,
      metadataType: null,
      checksums: this._getCheckums(archiveMetadata),
      available: null,
    } as ArchiveDataFile;
  }

  private static _getCheckums(archiveMetadata: ArchiveMetadataDataFile): Checksum[] {
    const checksums: Checksum[] = [];

    if (isTruthyObject(archiveMetadata.metadata?.checksums)) {
      MappingObjectUtil.forEach(archiveMetadata.metadata.checksums as any, (value: string, key, mappingObject) => {
        checksums.push({
          checksum: value,
          checksumAlgo: this.getAlgoName(key),
        });
      });
    }

    return checksums;
  }

  static getAlgoName(algoNameInArchive: string): string {
    switch (algoNameInArchive) {
      case MetadataDataFileEnum.checksumsSha1:
        return "SHA1";
      case MetadataDataFileEnum.checksumsSha256:
        return "SHA256";
      case "MD5":
      default:
        return algoNameInArchive;
    }
  }
}
