import {Injectable} from "@angular/core";
import {
  ActivatedRouteSnapshot,
  CanActivate,
  Router,
} from "@angular/router";
import {AppState} from "@app/stores/app.state";
import {Navigate} from "@ngxs/router-plugin";
import {Store} from "@ngxs/store";
import {
  AppRoutesEnum,
  RoutesEnum,
} from "@shared/enums/routes.enum";
import {TourRouteIdEnum} from "@shared/enums/tour-route-id.enum";
import {SecurityService} from "@shared/services/security.service";
import {
  Observable,
  of,
} from "rxjs";
import {
  ApiService,
  MemoizedUtil,
} from "solidify-frontend";

@Injectable({
  providedIn: "root",
})
export class DepositRoleGuardDetailService implements CanActivate {
  constructor(public router: Router,
              public store: Store,
              public apiService: ApiService,
              private readonly _securityService: SecurityService) {
  }

  canActivate(route: ActivatedRouteSnapshot): Observable<boolean> {
    const depositId: string = route.params[AppRoutesEnum.paramIdWithoutPrefixParam];
    const isInTourMode = MemoizedUtil.selectSnapshot(this.store, AppState, state => state.isInTourMode);
    const isDepositIdForTour = depositId === TourRouteIdEnum.tourDepositId;
    if (isDepositIdForTour || isInTourMode) {
      const isAuthorizedToDisplayTour = isDepositIdForTour && isInTourMode;
      if (!isAuthorizedToDisplayTour) {
        this.store.dispatch(new Navigate([RoutesEnum.deposit]));
      }
      return of(isAuthorizedToDisplayTour);
    }
    return this._securityService.canSeeDetailDepositById(depositId);
  }
}
