import {Injectable} from "@angular/core";
import {
  ActivatedRouteSnapshot,
  CanActivate,
  Router,
} from "@angular/router";
import {HomeAction} from "@home/stores/home.action";
import {HomeState} from "@home/stores/home.state";
import {Navigate} from "@ngxs/router-plugin";
import {
  Actions,
  ofActionCompleted,
  Store,
} from "@ngxs/store";
import {
  AppRoutesEnum,
  RoutesEnum,
} from "@shared/enums/routes.enum";
import {Observable} from "rxjs";
import {map} from "rxjs/operators";
import {
  ApiService,
  isNullOrUndefined,
  MemoizedUtil,
  StoreUtil,
} from "solidify-frontend";

@Injectable({
  providedIn: "root",
})
export class ArchiveGuardService implements CanActivate {
  constructor(public router: Router,
              private _store: Store,
              public apiService: ApiService,
              private _actions$: Actions) {
  }

  canActivate(route: ActivatedRouteSnapshot): Observable<boolean> {
    const archiveId: string = route.params[AppRoutesEnum.paramIdWithoutPrefixParam];

    return StoreUtil.dispatchSequentialActionAndWaitForSubActionsCompletion(this._store, [
      {
        action: new HomeAction.SearchDetail(archiveId),
        subActionCompletions: [
          this._actions$.pipe(ofActionCompleted(HomeAction.SearchDetailSuccess)),
          this._actions$.pipe(ofActionCompleted(HomeAction.SearchDetailFail)),
        ],
      },
    ]).pipe(
      map(result => {
        if (result === false) {
          return this._redirectToPageNotFound(archiveId);
        }

        const archive = MemoizedUtil.selectSnapshot(this._store, HomeState, state => state.current);

        if (isNullOrUndefined(archive) || archive.resId !== archiveId) {
          return this._redirectToPageNotFound(archiveId);
        }
        return true;
      }),
    );
  }

  private _redirectToPageNotFound(id: string): false {
    this._store.dispatch(new Navigate([RoutesEnum.homeNotFound, id]));
    return false;
  }
}
