import {Injectable} from "@angular/core";
import {
  ActivatedRouteSnapshot,
  CanActivate,
  Router,
  RouterStateSnapshot,
} from "@angular/router";
import {DlcmData} from "@app/shared/models/dlcm-route.model";
import {LocalStateModel} from "@app/shared/models/local-state.model";
import {PermissionUtil} from "@app/shared/utils/permission.util";
import {AppState} from "@app/stores/app.state";
import {Enums} from "@enums";
import {Store} from "@ngxs/store";
import {AppRoutesEnum} from "@shared/enums/routes.enum";
import {PublicUrlHelper} from "@shared/helpers/public-url.helper";
import {UrlQueryParamHelper} from "@shared/helpers/url-query-param.helper";
import {isNotNullOrUndefined} from "codelyzer/util/isNotNullOrUndefined";
import {Observable} from "rxjs";
import {
  filter,
  map,
  take,
} from "rxjs/operators";
import {
  isEmptyString,
  isNullOrUndefined,
  MappingObject,
  MappingObjectUtil,
  MemoizedUtil,
  OAuth2Service,
} from "solidify-frontend";

@Injectable({
  providedIn: "root",
})
export class ApplicationRoleGuardService implements CanActivate {
  constructor(public router: Router, public store: Store, private readonly _oauthService: OAuth2Service) {
  }

  canActivate(route: ActivatedRouteSnapshot, state1: RouterStateSnapshot): Observable<boolean> {
    const data = route.data as DlcmData;
    const permission = isNullOrUndefined(data) ? undefined : data.permission;
    return MemoizedUtil.select(this.store, AppState, st => st.isApplicationInitialized).pipe(
      filter(isApplicationInitialized => isApplicationInitialized),
      take(1),
      map(() => {
        const currentUserRole = this.store.selectSnapshot<Enums.UserApplicationRole.UserApplicationRoleEnum[]>((st: LocalStateModel) => st.application.userRoles);
        const isLoggedIn = this.store.selectSnapshot<boolean>((st: LocalStateModel) => st.application.isLoggedIn);
        const isAuthorized = PermissionUtil.isUserHavePermission(isLoggedIn, permission, currentUserRole);
        if (isAuthorized) {
          const stateValue: string = UrlQueryParamHelper.getStateFromUrlWithQueryParamOAuth2(state1.url);
          if (!isNullOrUndefined(stateValue) && !isEmptyString(stateValue)) {
            const queryParams = UrlQueryParamHelper.getListQueryParams(state1.url);
            const queryParamMapping: MappingObject = {};
            if (!isNullOrUndefined(queryParams)) {
              queryParams.forEach(query => {
                const keyValue = query.split(UrlQueryParamHelper.getQueryParamValueSeparator());
                MappingObjectUtil.set(queryParamMapping, keyValue[0], keyValue[1]);
              });
            }
            this.router.navigate([stateValue], {queryParams: queryParamMapping});
          } else {
            if (route.queryParamMap.has(UrlQueryParamHelper.QUERY_PARAM_OAUTH_CODE) || route.queryParamMap.has(UrlQueryParamHelper.QUERY_PARAM_OAUTH_STATE)) {
              const queryParamCleaned = Object.assign({}, route.queryParams);
              MappingObjectUtil.delete(queryParamCleaned, UrlQueryParamHelper.QUERY_PARAM_OAUTH_CODE);
              MappingObjectUtil.delete(queryParamCleaned, UrlQueryParamHelper.QUERY_PARAM_OAUTH_STATE);
              this.router.navigate([AppRoutesEnum.home], {queryParams: queryParamCleaned});
            }
          }
          return true;
        }
        if (!isLoggedIn) {
          // if the url is the base url, index or anything under #home, do nothing but if contains path, log-in and redirect to the specific url
          const isPublicUrl = PublicUrlHelper.listPublicUrls.find(url => state1.url.startsWith(url));
          if (isNotNullOrUndefined(isPublicUrl)) {
            this._oauthService.initAuthorizationCodeFlow(state1.url);
          }
        }
        this.router.navigate([AppRoutesEnum.home]);
        return false;
      }),
    );
  }
}
