import {OverlayModule} from "@angular/cdk/overlay";
import {HttpClientTestingModule} from "@angular/common/http/testing";
import {TestBed} from "@angular/core/testing";
import {MatSnackBar} from "@angular/material/snack-bar";
import {
  ActivatedRouteSnapshot,
  Router,
} from "@angular/router";
import {appModuleState} from "@app/app.module";
import {ApplicationRolePermissionEnum} from "@app/shared/enums/application-role-permission.enum";
import {ApplicationRoleGuardService} from "@app/shared/guards/application-role-guard.service";
import {DlcmData} from "@app/shared/models/dlcm-route.model";
import {AppAction} from "@app/stores/app.action";
import {
  AppState,
  AppStateModel,
} from "@app/stores/app.state";
import {Enums} from "@enums";
import {environment} from "@environments/environment";
import {TranslateService} from "@ngx-translate/core";
import {
  NgxsModule,
  Store,
} from "@ngxs/store";
import {JwtTokenHelper} from "@shared/helpers/jwt-token.helper";
import {Token} from "@shared/models/token.model";
import {
  ENVIRONMENT,
  SNACK_BAR,
} from "solidify-frontend";

describe("ApplicationRoleGuardService", () => {
  const router = {
    navigate: jasmine.createSpy("navigate"),
    parseUrl: jasmine.createSpy("parseUrl").and.returnValue(false),
  };
  const mockTranslateService = jasmine.createSpyObj("TranslateService", ["getTranslation", "setDefaultLang", "use"]);

  beforeEach(() => {
    TestBed.configureTestingModule({
      imports: [HttpClientTestingModule, NgxsModule.forRoot([
        ...appModuleState,
      ]), OverlayModule],
      providers: [
        {
          provide: ApplicationRoleGuardService,
          useClass: ApplicationRoleGuardService,
        },
        {
          provide: TranslateService,
          useValue: mockTranslateService,
        },
        {
          provide: Router,
          useValue: router,
        },
        {
          provide: ENVIRONMENT,
          useValue: environment,
        },
        {
          provide: SNACK_BAR,
          useClass: MatSnackBar,
        },
      ],
    });
  });

  async function markAppAsInitialized(store: Store): Promise<void> {
    await store.dispatch(new AppAction.InitApplicationSuccess(null)).toPromise();
    const state = store.selectSnapshot(AppState);
    expect(state.isApplicationInitialized).toBe(true);
  }

  function checkCanHitRoute(data: DlcmData, canHit: boolean): void {
    const service: ApplicationRoleGuardService = TestBed.get(ApplicationRoleGuardService);
    service.canActivate({data: data} as ActivatedRouteSnapshot, {url: undefined} as any).subscribe(res => expect(res).toBe(canHit));
  }

  it("be able to hit route when user is logged in", async () => {
    const store: Store = TestBed.get(Store);

    store.dispatch(new AppAction.LoginSuccess(null));
    const state = store.selectSnapshot(AppState);
    expect(state.isLoggedIn).toBe(true);

    await markAppAsInitialized(store);
    checkCanHitRoute({permission: ApplicationRolePermissionEnum.noPermission}, true);
  });

  it("not be able to hit route when user is not logged in", async () => {
    const store: Store = TestBed.get(Store);

    await store.dispatch(new AppAction.LoginFail(null)).toPromise();
    const state = store.selectSnapshot(AppState);
    expect(state.isLoggedIn).toBe(false);

    await markAppAsInitialized(store);
    checkCanHitRoute({permission: ApplicationRolePermissionEnum.userPermission}, false);
  });

  it("should be created", () => {
    const service: ApplicationRoleGuardService = TestBed.get(ApplicationRoleGuardService);
    expect(service).toBeTruthy();
  });

  it("be able to hit route for permission user when user have role [USER]", async () => {
    const store: Store = TestBed.get(Store);
    const simulatingCurrentUserRole = [Enums.UserApplicationRole.UserApplicationRoleEnum.user];
    spyOn(JwtTokenHelper, "decodeToken").and.returnValue({authorities: simulatingCurrentUserRole} as Token);

    store.dispatch(new AppAction.LoginSuccess(null));
    const state: AppStateModel = store.selectSnapshot(AppState);
    expect(state.isLoggedIn).toBe(true);
    expect(state.userRoles).toEqual(simulatingCurrentUserRole);

    await markAppAsInitialized(store);
    checkCanHitRoute({permission: ApplicationRolePermissionEnum.userPermission}, true);
  });

  it("be able to hit route for permission admin when user have role [ADMIN]", async () => {
    const store: Store = TestBed.get(Store);
    const simulatingCurrentUserRole = [Enums.UserApplicationRole.UserApplicationRoleEnum.admin];
    spyOn(JwtTokenHelper, "decodeToken").and.returnValue({authorities: simulatingCurrentUserRole} as Token);

    store.dispatch(new AppAction.LoginSuccess(null));
    const state: AppStateModel = store.selectSnapshot(AppState);
    expect(state.isLoggedIn).toBe(true);
    expect(state.userRoles).toEqual(simulatingCurrentUserRole);

    await markAppAsInitialized(store);
    checkCanHitRoute({permission: ApplicationRolePermissionEnum.adminPermission}, true);
  });

  it("be able to hit route for permission user when user have role [ADMIN]", async () => {
    const store: Store = TestBed.get(Store);
    const simulatingCurrentUserRole = [Enums.UserApplicationRole.UserApplicationRoleEnum.admin];
    spyOn(JwtTokenHelper, "decodeToken").and.returnValue({authorities: simulatingCurrentUserRole} as Token);

    store.dispatch(new AppAction.LoginSuccess(null));
    const state: AppStateModel = store.selectSnapshot(AppState);
    expect(state.isLoggedIn).toBe(true);
    expect(state.userRoles).toEqual(simulatingCurrentUserRole);

    await markAppAsInitialized(store);
    checkCanHitRoute({permission: ApplicationRolePermissionEnum.userPermission}, true);
  });

  it("not be able to hit route for permission root when user have role [ADMIN]", async () => {
    const store: Store = TestBed.get(Store);
    const simulatingCurrentUserRole = [Enums.UserApplicationRole.UserApplicationRoleEnum.admin];
    spyOn(JwtTokenHelper, "decodeToken").and.returnValue({authorities: simulatingCurrentUserRole} as Token);

    store.dispatch(new AppAction.LoginSuccess(null));
    const state: AppStateModel = store.selectSnapshot(AppState);
    expect(state.isLoggedIn).toBe(true);
    expect(state.userRoles).toEqual(simulatingCurrentUserRole);

    await markAppAsInitialized(store);
    checkCanHitRoute({permission: ApplicationRolePermissionEnum.rootPermission}, false);
  });

  it("be able to hit route for permission admin when user have role [USER, ADMIN]", async () => {
    const store: Store = TestBed.get(Store);
    const simulatingCurrentUserRole = [Enums.UserApplicationRole.UserApplicationRoleEnum.user, Enums.UserApplicationRole.UserApplicationRoleEnum.admin];
    spyOn(JwtTokenHelper, "decodeToken").and.returnValue({authorities: simulatingCurrentUserRole} as Token);

    store.dispatch(new AppAction.LoginSuccess(null));
    const state: AppStateModel = store.selectSnapshot(AppState);
    expect(state.isLoggedIn).toBe(true);
    expect(state.userRoles).toEqual(simulatingCurrentUserRole);

    await markAppAsInitialized(store);
    checkCanHitRoute({permission: ApplicationRolePermissionEnum.adminPermission}, true);
  });

  it("not be able to hit route for permission admin when user have role [USER]", async () => {
    const service: ApplicationRoleGuardService = TestBed.get(ApplicationRoleGuardService);
    const store: Store = TestBed.get(Store);
    const simulatingCurrentUserRole = [Enums.UserApplicationRole.UserApplicationRoleEnum.user];
    spyOn(JwtTokenHelper, "decodeToken").and.returnValue({authorities: simulatingCurrentUserRole} as Token);

    store.dispatch(new AppAction.LoginSuccess(null));
    const state: AppStateModel = store.selectSnapshot(AppState);
    expect(state.isLoggedIn).toBe(true);
    expect(state.userRoles).toEqual(simulatingCurrentUserRole);

    await markAppAsInitialized(store);
    checkCanHitRoute({permission: ApplicationRolePermissionEnum.adminPermission}, false);
  });

  it("not be able to hit route for permission root when user have role [USER, ADMIN]", async () => {
    const service: ApplicationRoleGuardService = TestBed.get(ApplicationRoleGuardService);
    const store: Store = TestBed.get(Store);
    const simulatingCurrentUserRole = [Enums.UserApplicationRole.UserApplicationRoleEnum.user, Enums.UserApplicationRole.UserApplicationRoleEnum.admin];
    spyOn(JwtTokenHelper, "decodeToken").and.returnValue({authorities: simulatingCurrentUserRole} as Token);

    store.dispatch(new AppAction.LoginSuccess(null));
    const state: AppStateModel = store.selectSnapshot(AppState);
    expect(state.isLoggedIn).toBe(true);
    expect(state.userRoles).toEqual(simulatingCurrentUserRole);

    await markAppAsInitialized(store);
    checkCanHitRoute({permission: ApplicationRolePermissionEnum.rootPermission}, false);
  });

  it("be able to hit route for permission root when user have role [USER, ADMIN, ROOT]", async () => {
    const service: ApplicationRoleGuardService = TestBed.get(ApplicationRoleGuardService);
    const store: Store = TestBed.get(Store);
    const simulatingCurrentUserRole = [Enums.UserApplicationRole.UserApplicationRoleEnum.user, Enums.UserApplicationRole.UserApplicationRoleEnum.admin, Enums.UserApplicationRole.UserApplicationRoleEnum.root];
    spyOn(JwtTokenHelper, "decodeToken").and.returnValue({authorities: simulatingCurrentUserRole} as Token);

    store.dispatch(new AppAction.LoginSuccess(null));
    const state: AppStateModel = store.selectSnapshot(AppState);
    expect(state.isLoggedIn).toBe(true);
    expect(state.userRoles).toEqual(simulatingCurrentUserRole);

    await markAppAsInitialized(store);
    checkCanHitRoute({permission: ApplicationRolePermissionEnum.rootPermission}, true);
  });

  it("be able to hit route without permission when user have no role", async () => {
    const store: Store = TestBed.get(Store);
    const simulatingCurrentUserRole = null;
    spyOn(JwtTokenHelper, "decodeToken").and.returnValue({authorities: simulatingCurrentUserRole} as Token);

    store.dispatch(new AppAction.LoginSuccess(null));
    const state: AppStateModel = store.selectSnapshot(AppState);
    expect(state.isLoggedIn).toBe(true);
    expect(state.userRoles).toEqual(simulatingCurrentUserRole);

    await markAppAsInitialized(store);
    checkCanHitRoute({}, true);
  });

  it("be able to hit route without permission when user have no role", async () => {
    const store: Store = TestBed.get(Store);
    const simulatingCurrentUserRole = null;
    spyOn(JwtTokenHelper, "decodeToken").and.returnValue({authorities: simulatingCurrentUserRole} as Token);

    store.dispatch(new AppAction.LoginSuccess(null));
    const state: AppStateModel = store.selectSnapshot(AppState);
    expect(state.isLoggedIn).toBe(true);
    expect(state.userRoles).toEqual(simulatingCurrentUserRole);

    await markAppAsInitialized(store);
    checkCanHitRoute({permission: ApplicationRolePermissionEnum.noPermission}, true);
  });

  it("not be able to hit route for permission user when user have no role", async () => {
    const store: Store = TestBed.get(Store);
    const simulatingCurrentUserRole = null;
    spyOn(JwtTokenHelper, "decodeToken").and.returnValue({authorities: simulatingCurrentUserRole} as Token);

    store.dispatch(new AppAction.LoginSuccess(null));
    const state: AppStateModel = store.selectSnapshot(AppState);
    expect(state.isLoggedIn).toBe(true);
    expect(state.userRoles).toEqual(simulatingCurrentUserRole);

    await markAppAsInitialized(store);
    checkCanHitRoute({permission: ApplicationRolePermissionEnum.userPermission}, false);
  });

  it("not be able to hit route for permission user when user have empty role", async () => {
    const store: Store = TestBed.get(Store);
    const simulatingCurrentUserRole = [];
    spyOn(JwtTokenHelper, "decodeToken").and.returnValue({authorities: simulatingCurrentUserRole} as Token);

    store.dispatch(new AppAction.LoginSuccess(null));
    const state: AppStateModel = store.selectSnapshot(AppState);
    expect(state.isLoggedIn).toBe(true);
    expect(state.userRoles).toEqual(simulatingCurrentUserRole);

    await markAppAsInitialized(store);
    checkCanHitRoute({permission: ApplicationRolePermissionEnum.userPermission}, false);
  });

  it("not be able to hit route for permission admin when user have no role", async () => {
    const store: Store = TestBed.get(Store);
    const simulatingCurrentUserRole = null;
    spyOn(JwtTokenHelper, "decodeToken").and.returnValue({authorities: simulatingCurrentUserRole} as Token);

    store.dispatch(new AppAction.LoginSuccess(null));
    const state: AppStateModel = store.selectSnapshot(AppState);
    expect(state.isLoggedIn).toBe(true);
    expect(state.userRoles).toEqual(simulatingCurrentUserRole);

    await markAppAsInitialized(store);
    checkCanHitRoute({permission: ApplicationRolePermissionEnum.adminPermission}, false);
  });

  it("not be able to hit route for permission admin when user have empty role", async () => {
    const store: Store = TestBed.get(Store);
    const simulatingCurrentUserRole = [];
    spyOn(JwtTokenHelper, "decodeToken").and.returnValue({authorities: simulatingCurrentUserRole} as Token);

    store.dispatch(new AppAction.LoginSuccess(null));
    const state: AppStateModel = store.selectSnapshot(AppState);
    expect(state.isLoggedIn).toBe(true);
    expect(state.userRoles).toEqual(simulatingCurrentUserRole);

    await markAppAsInitialized(store);
    checkCanHitRoute({permission: ApplicationRolePermissionEnum.adminPermission}, false);
  });

  it("be able to hit route for no permission when user have empty role", async () => {
    const store: Store = TestBed.get(Store);
    const simulatingCurrentUserRole = [];
    spyOn(JwtTokenHelper, "decodeToken").and.returnValue({authorities: simulatingCurrentUserRole} as Token);

    store.dispatch(new AppAction.LoginSuccess(null));
    const state: AppStateModel = store.selectSnapshot(AppState);
    expect(state.isLoggedIn).toBe(true);
    expect(state.userRoles).toEqual(simulatingCurrentUserRole);

    await markAppAsInitialized(store);
    checkCanHitRoute({permission: ApplicationRolePermissionEnum.noPermission}, true);
  });
});
