import {Injectable} from "@angular/core";
import {
  ActivatedRouteSnapshot,
  CanActivate,
  Router,
} from "@angular/router";
import {AppState} from "@app/stores/app.state";
import {Enums} from "@enums";
import {environment} from "@environments/environment";
import {Navigate} from "@ngxs/router-plugin";
import {Store} from "@ngxs/store";
import {RoutesEnum} from "@shared/enums/routes.enum";
import {LocalStateModel} from "@shared/models/local-state.model";
import {PermissionUtil} from "@shared/utils/permission.util";
import {
  Observable,
  of,
} from "rxjs";
import {
  filter,
  map,
  take,
} from "rxjs/operators";
import {
  isFalse,
  isTrue,
  MemoizedUtil,
} from "solidify-frontend";

@Injectable({
  providedIn: "root",
})
export class ApplicationMaintenanceGuardService implements CanActivate {
  constructor(public _router: Router,
              public _store: Store) {
  }

  canActivate(route: ActivatedRouteSnapshot): Observable<boolean> {
    if (isTrue(MemoizedUtil.selectSnapshot(this._store, AppState, state => state.isServerOffline))) {
      this._store.dispatch(new Navigate([RoutesEnum.serverOffline]));
    }

    if (isFalse(environment.maintenanceMode)) {
      return of(true);
    }

    return MemoizedUtil.select(this._store, AppState, st => st.isApplicationInitialized).pipe(
      filter(isApplicationInitialized => isApplicationInitialized),
      take(1),
      map(() => {
        const currentUserRole = this._store.selectSnapshot<Enums.UserApplicationRole.UserApplicationRoleEnum[]>((st: LocalStateModel) => st.application.userRoles);
        const isLoggedIn = this._store.selectSnapshot<boolean>((st: LocalStateModel) => st.application.isLoggedIn);

        if (isLoggedIn === false || !PermissionUtil.isRootPermission(currentUserRole)) {
          this._store.dispatch(new Navigate([RoutesEnum.maintenance]));
          return false;
        }
        return true;
      }),
    );
  }
}
