import {environment} from "@environments/environment";
import {FileInput} from "@shared/filevisualizer/models/file-info.model";
import {AbstractFileVisualizer} from "@shared/filevisualizer/services/abstract-file-visualizer.service";
import {isTruthyObject} from "solidify-frontend";

export class VideoClassicalFileVisualizerService extends AbstractFileVisualizer {

  type: string = "videoClassicalPlugin";

  video: HTMLVideoElement = null;

  canHandle(fileInfo: FileInput): boolean {
    if (!isTruthyObject(fileInfo)) {
      return false;
    } else {
      return this.canHandleByExtension(fileInfo) || this.canHandleByMimeTypeContentType(fileInfo) || this.canHandleByPuid(fileInfo);
    }
  }

  canHandleByExtension(fileInfo: FileInput): boolean {
    if (!isTruthyObject(fileInfo)) {
      return false;
    } else {
      return environment.visualizationClassicalMovieExtensions.some(value => value === fileInfo.fileExtension);
    }
  }

  isVisualizationOnGoing(fileInfo: FileInput, domElement: Element): boolean {
    return isTruthyObject(this.video);
  }

  canHandleWithSize(fileInfo: FileInput): boolean {
    if (!isTruthyObject(fileInfo)) {
      return false;
    } else {
      return fileInfo.dataFile.fileSize <= environment.visualizationMaxSizeFile;
    }
  }

  canHandleByMimeTypeContentType(fileInfo: FileInput): boolean {
    if (!isTruthyObject(fileInfo)) {
      return false;
    } else {
      return environment.visualizationClassicalMovieContentType.some(value => value === fileInfo.dataFile.fileFormat?.contentType) || environment.visualizationClassicalMovieMimeType.some(value => value === fileInfo.dataFile.fileFormat.contentType);

    }
  }

  canHandleByPuid(fileInfo: FileInput): boolean {
    if (!isTruthyObject(fileInfo)) {
      return false;
    } else {
      return environment.visualizationClassicalMoviePronomId.some(value => value === fileInfo.dataFile.fileFormat?.puid);
    }
  }

  closeVisualizer(fileInfo: FileInput, domElement: Element): void {
    domElement.remove();
    this.video = null;
  }

  openVisualizer(fileInfo: FileInput, domElement: Element): void {
    const url = URL.createObjectURL(fileInfo.blob);
    this.video = document.createElement("video");
    this.video.style.backgroundColor = "black";
    this.video.controls = true;
    this.video.autoplay = true;

    const source = document.createElement("source");
    source.src = url;
    source.type = fileInfo.dataFile.fileFormat?.contentType;
    this.video.appendChild(source);
    this.autoResizeElement();
    domElement.appendChild(this.video);
  }

  autoResizeElement(): void {
    this.video.style.maxHeight = "100%";
    this.video.style.maxWidth = "100%";
  }

  doAction(fileInfo: FileInput, domElement: Element): void {
  }

  isZoomSupported(): boolean {
    return false;
  }

  handleZoom(zoomEnable: boolean, fileInfo: FileInput, domElement: Element): void {
  }

  isFullScreenSupported(): boolean {
    return false;
  }

}
