import {Injectable} from "@angular/core";
import {environment} from "@environments/environment";
import {FileInput} from "@shared/filevisualizer/models/file-info.model";
import {AbstractFileVisualizer} from "@shared/filevisualizer/services/abstract-file-visualizer.service";
import {HighlightJS} from "ngx-highlightjs";
import {tap} from "rxjs/operators";
import {isTruthyObject} from "solidify-frontend";

@Injectable()
export class TextFileVisualizerService extends AbstractFileVisualizer {

  type: string = "textPlugin";
  pre: HTMLPreElement = null;

  constructor(private readonly hilight: HighlightJS) {
    super();
  }

  canHandle(fileInfo: FileInput): boolean {
    if (!isTruthyObject(fileInfo)) {
      return false;
    } else {
      return this.canHandleByExtension(fileInfo) || this.canHandleByMimeTypeContentType(fileInfo) || this.canHandleByPuid(fileInfo);
    }
  }

  canHandleByExtension(fileInfo: FileInput): boolean {
    if (!isTruthyObject(fileInfo)) {
      return false;
    } else {
      return environment.visualizationTextFileExtension.some(value => value === fileInfo.fileExtension);
    }
  }

  isVisualizationOnGoing(fileInfo: FileInput, domElement: Element): boolean {
    return isTruthyObject(this.pre);
  }

  canHandleWithSize(fileInfo: FileInput): boolean {
    if (!isTruthyObject(fileInfo)) {
      return false;
    } else {
      return fileInfo.dataFile.fileSize <= environment.visualizationMaxSizeFile;
    }
  }

  canHandleByMimeTypeContentType(fileInfo: FileInput): boolean {
    if (!isTruthyObject(fileInfo)) {
      return false;
    } else {
      return environment.visualizationTextFileContentType.some(value => value === fileInfo.dataFile.fileFormat?.contentType) || environment.visualizationTextFileMimeType.some(value => value === fileInfo.dataFile.fileFormat.contentType);
    }
  }

  canHandleByPuid(fileInfo: FileInput): boolean {
    if (!isTruthyObject(fileInfo)) {
      return false;
    } else {
      return environment.visualizationTextFilePronomId.some(value => value === fileInfo.dataFile.fileFormat?.puid);
    }
  }

  closeVisualizer(fileInfo: FileInput, domElement: Element): void {
    domElement.remove();
    this.pre = null;
  }

  openVisualizer(fileInfo: FileInput, domElement: Element): void {
    this.pre = document.createElement("pre");
    this.pre.style.display = "block";
    this.pre.style.lineHeight = "1.42857143";
    this.pre.style.fontSize = "13px";
    this.pre.style.margin = "0 0 10px";
    this.pre.style.padding = "9.5px";
    this.pre.style.color = "#333";
    this.pre.style.overflowWrap = "break-word";
    this.pre.style.wordWrap = "break-word";
    this.pre.style.whiteSpace = "break-spaces";
    this.pre.style.backgroundColor = "#f5f5f5";
    this.pre.style.border = "1px solid #ccc";
    this.pre.style.borderRadius = "4px";
    const fr = new FileReader();
    fr.onload = (
      (e) => {
        this.hilight.highlightAuto(fr.result as string, ["xml", "json", "java"]).pipe(
          tap(value => {
            this.pre.innerHTML = value.value;
          }),
        ).subscribe();
        this.hilight.highlightBlock(this.pre);
      }
    );
    fr.readAsText(fileInfo.blob, "UTF-8");
    domElement.appendChild(this.pre);
  }

  doAction(fileInfo: FileInput, domElement: Element): void {
  }

  isZoomSupported(): boolean {
    return false;
  }

  handleZoom(zoomEnable: boolean, fileInfo: FileInput, domElement: Element): void {
  }

  isFullScreenSupported(): boolean {
    return true;
  }
}
