import {environment} from "@environments/environment";
import {FileInput} from "@shared/filevisualizer/models/file-info.model";
import {AbstractFileVisualizer} from "@shared/filevisualizer/services/abstract-file-visualizer.service";
import {
  isNullOrUndefined,
  isTruthyObject,
} from "solidify-frontend";
import WaveSurfer from "wavesurfer.js";

export class SoundFileVisualizerService extends AbstractFileVisualizer {

  type: string = "soundPlugin";

  wave: WaveSurfer = null;

  canHandle(fileInfo: FileInput): boolean {
    if (!isTruthyObject(fileInfo)) {
      return false;
    } else {
      return this.canHandleByExtension(fileInfo) || this.canHandleByMimeTypeContentType(fileInfo) || this.canHandleByPuid(fileInfo);
    }
  }

  canHandleByExtension(fileInfo: FileInput): boolean {
    if (!isTruthyObject(fileInfo)) {
      return false;
    } else {
      return environment.visualizationSoundExtension.some(value => value === fileInfo.fileExtension);
    }
  }

  canHandleWithSize(fileInfo: FileInput): boolean {
    if (!isTruthyObject(fileInfo)) {
      return false;
    } else {
      return fileInfo.dataFile.fileSize <= environment.visualizationMaxSizeFile;
    }
  }

  isVisualizationOnGoing(fileInfo: FileInput, domElement: Element): boolean {
    return isTruthyObject(this.wave);
  }

  canHandleByMimeTypeContentType(fileInfo: FileInput): boolean {
    if (!isTruthyObject(fileInfo)) {
      return false;
    } else {
      return environment.visualizationSoundContentType.some(value => value === fileInfo.dataFile.fileFormat?.contentType) || environment.visualizationSoundMimeType.some(value => value === fileInfo.dataFile.fileFormat.contentType);
    }
  }

  canHandleByPuid(fileInfo: FileInput): boolean {
    if (!isTruthyObject(fileInfo)) {
      return false;
    } else {
      return environment.visualizationSoundPronomId.some(value => value === fileInfo.dataFile.fileFormat.puid);
    }
  }

  closeVisualizer(fileInfo: FileInput, domElement: Element): void {
    this.wave.unAll();
    this.wave.stop();
    this.wave.destroy();
  }

  openVisualizer(fileInfo: FileInput, domElement: Element): void {
    if (this.canHandle(fileInfo)) {
      const divContainer = document.createElement("div");
      divContainer.style.display = "grid";
      divContainer.style.gridTemplateRows = "auto auto auto";
      const waveContainer = document.createElement("div");
      waveContainer.id = "wave";
      waveContainer.style.overflow = "hidden";
      divContainer.appendChild(waveContainer);
      const timelineContainer = document.createElement("div");
      timelineContainer.style.display = "flex";
      timelineContainer.style.justifyContent = "center";
      const timeElement = document.createElement("span");
      timeElement.id = "time";
      timelineContainer.appendChild(timeElement);
      const buttonContainer = document.createElement("div");
      buttonContainer.style.display = "flex";
      buttonContainer.style.alignItems = "center";
      buttonContainer.style.justifyContent = "center";
      const matButton = document.createElement("button");
      matButton.onclick = (ev => this.wave.playPause());
      matButton.textContent = "Play | Pause";
      matButton.classList.add("mat-button");
      matButton.classList.add("mat-focus-indicator");
      matButton.classList.add("mat-button-base");
      matButton.classList.add("mat-primary");
      buttonContainer.appendChild(matButton);
      divContainer.appendChild(buttonContainer);
      divContainer.appendChild(timelineContainer);
      document.createElement("div");
      domElement.appendChild(divContainer);
      this.wave = WaveSurfer.create({
        container: document.querySelector("#wave"),
        barWidth: 2,
        barHeight: 1,
        overflow: "hidden ! important",
        waveColor: "violet",
        progressColor: "purple",
      });

      const url = URL.createObjectURL(fileInfo.blob);
      this.wave.load(url);
      this.wave.on("ready", () => {
        this.wave.play();
      });

      this.wave.on("audioprocess", () => {
        const totalTime = this.wave.getDuration();
        const currentTime = this.wave.getCurrentTime();
        const timeElem = document.getElementById("time");
        if (isNullOrUndefined(timeElem)) {
          return;
        }
        timeElem.innerText = this.getMiliSecondeDisplay(Math.round(currentTime * 1000)) + "/" + this.getMiliSecondeDisplay(Math.round(totalTime * 1000));
      });

      this.wave.on("pause", () => {
        const totalTime = this.wave.getDuration();
        const currentTime = this.wave.getCurrentTime();
        const timeElem = document.getElementById("time");
        if (isNullOrUndefined(timeElem)) {
          return;
        }
        timeElem.innerText = this.getMiliSecondeDisplay(Math.round(currentTime * 1000)) + "/" + this.getMiliSecondeDisplay(Math.round(totalTime * 1000));
      });
    }
  }

  private getMiliSecondeDisplay(millisec: number): string {
    let seconds: any = (millisec / 1000).toFixed(0);
    let minutes: any = Math.floor(seconds / 60);
    let hours: any = "";
    if (minutes > 59) {
      hours = Math.floor(minutes / 60);
      hours = (hours >= 10) ? hours : "0" + hours;
      minutes = minutes - (hours * 60);
      minutes = (minutes >= 10) ? minutes : "0" + minutes;
    }
    seconds = Math.floor(seconds % 60);
    seconds = (seconds >= 10) ? seconds : "0" + seconds;
    if (hours !== "") {
      return hours + ":" + minutes + ":" + seconds;
    }
    return minutes + ":" + seconds;
  }

  doAction(fileInfo: FileInput, domElement: Element): void {
  }

  isZoomSupported(): boolean {
    return false;
  }

  handleZoom(zoomEnable: boolean, fileInfo: FileInput, domElement: Element): void {
  }

  isFullScreenSupported(): boolean {
    return false;
  }
}
