import {
  ComponentFactoryResolver,
  ComponentRef,
  Injectable,
  NgZone,
} from "@angular/core";
import {DomSanitizer} from "@angular/platform-browser";
import {environment} from "@environments/environment";
import {AddExternalComponentDirective} from "@shared/filevisualizer/directives/add-external-component.directive";
import {FileInput} from "@shared/filevisualizer/models/file-info.model";
import {AbstractFileVisualizer} from "@shared/filevisualizer/services/abstract-file-visualizer.service";
import {PdfViewerComponent} from "ng2-pdf-viewer";
import {isTruthyObject} from "solidify-frontend";

@Injectable()
export class PdfFileVisualizerService extends AbstractFileVisualizer {

  type: string = "pdfPlugin";

  private _componentRef: ComponentRef<PdfViewerComponent>;
  private _pdfViewerComponent: PdfViewerComponent;

  constructor(private readonly componentFactoryResolver: ComponentFactoryResolver,
              private readonly _ngZone: NgZone,
              private readonly _domSanitizer: DomSanitizer) {
    super();
  }

  canHandle(fileInfo: FileInput): boolean {
    if (!isTruthyObject(fileInfo)) {
      return false;
    } else {
      return this.canHandleByExtension(fileInfo) || this.canHandleByMimeTypeContentType(fileInfo) || this.canHandleByPuid(fileInfo);
    }
  }

  canHandleByExtension(fileInfo: FileInput): boolean {
    if (!isTruthyObject(fileInfo)) {
      return false;
    } else {
      return environment.visualizationPdfExtensions.some(value => value === fileInfo.fileExtension);
    }
  }

  canHandleWithSize(fileInfo: FileInput): boolean {
    if (!isTruthyObject(fileInfo)) {
      return false;
    } else {
      return fileInfo.dataFile.fileSize <= environment.visualizationMaxSizeFile;
    }
  }

  canHandleByMimeTypeContentType(fileInfo: FileInput): boolean {
    if (!isTruthyObject(fileInfo)) {
      return false;
    } else {
      return environment.visualizationPdfContentType.some(value => value === fileInfo.dataFile.fileFormat?.contentType) || environment.visualizationPdfMimeType.some(value => value === fileInfo.dataFile.fileFormat.contentType);
    }
  }

  isVisualizationOnGoing(fileInfo: FileInput, domElement: Element): boolean {
    return domElement.childNodes.item(0).nodeName === "PDF-VIEWER";
  }

  canHandleByPuid(fileInfo: FileInput): boolean {
    if (!isTruthyObject(fileInfo)) {
      return false;
    } else {
      return environment.visualizationPdfPronomId.some(value => value === fileInfo.dataFile.fileFormat.puid);
    }
  }

  closeVisualizer(fileInfo: FileInput, domElement: Element): void {
    domElement.remove();
  }

  openVisualizer(fileInfo: FileInput, domElement: Element, addExternalPlugin?: AddExternalComponentDirective): void {
    if (this.canHandle(fileInfo)) {
      const componentFactory = this.componentFactoryResolver.resolveComponentFactory(PdfViewerComponent);
      const addExternalPluginContainerRef = addExternalPlugin.viewContainerRef;
      addExternalPluginContainerRef.clear();
      this._componentRef = addExternalPluginContainerRef.createComponent(componentFactory);
      const url = URL.createObjectURL(fileInfo.blob);
      this._pdfViewerComponent = this._componentRef.instance as PdfViewerComponent;
      this._pdfViewerComponent.src = url;
      this._pdfViewerComponent.zoom = 1;
      this._pdfViewerComponent.autoresize = true;
      this._pdfViewerComponent.renderText = true;
      this._pdfViewerComponent.showBorders = true;
      this._pdfViewerComponent.fitToPage = true;
      this._pdfViewerComponent.originalSize = true;
      this._pdfViewerComponent.ngOnInit();
      this._pdfViewerComponent.showAll = true;
      this._componentRef.changeDetectorRef.detectChanges();
      domElement.classList.add("zoom-in");
    }
  }

  doAction(fileInfo: FileInput, domElement: Element): void {
    if (domElement.classList.contains("zoom-in")) {
      this._pdfViewerComponent.zoom = 2;
      this._componentRef.changeDetectorRef.detectChanges();
      this._pdfViewerComponent.updateSize();
      this._componentRef.changeDetectorRef.detectChanges();
      domElement.classList.remove("zoom-in");
      domElement.classList.add("zoom-out");
      return;
    }
    if (domElement.classList.contains("zoom-out")) {
      this._pdfViewerComponent.zoom = 1;
      this._componentRef.changeDetectorRef.detectChanges();
      this._pdfViewerComponent.updateSize();
      this._componentRef.changeDetectorRef.detectChanges();
      domElement.classList.remove("zoom-out");
      domElement.classList.add("zoom-in");
      return;
    }
  }

  isZoomSupported(): boolean {
    return false;
  }

  handleZoom(zoomEnable: boolean, fileInfo: FileInput, domElement: Element): void {
  }

  isFullScreenSupported(): boolean {
    return true;
  }

}
