import {
  ComponentFactoryResolver,
  ComponentRef,
  Injectable,
  NgZone,
} from "@angular/core";
import {DomSanitizer} from "@angular/platform-browser";
import {environment} from "@environments/environment";
import {AddExternalComponentDirective} from "@shared/filevisualizer/directives/add-external-component.directive";
import {FileInput} from "@shared/filevisualizer/models/file-info.model";
import {PdfViewerWithToolBarPresentational} from "@shared/filevisualizer/plugin/pdfviewerwithtoolbar/pdf-viewer-with-tool-bar.component";
import {AbstractFileVisualizer} from "@shared/filevisualizer/services/abstract-file-visualizer.service";
import {PdfViewerComponent} from "ng2-pdf-viewer";
import {
  CoreAbstractAngularElement,
  isTruthyObject,
} from "solidify-frontend";

@Injectable()
export class PdfFileVisualizerWithToolBarService extends CoreAbstractAngularElement implements AbstractFileVisualizer {

  type: string = "pdfWithToolBarPlugin";

  private _componentRef: ComponentRef<PdfViewerWithToolBarPresentational>;
  private _toolBarPdfViewerComponent: PdfViewerWithToolBarPresentational;

  private _pdfViewerComponentRef: ComponentRef<PdfViewerComponent>;
  private _pdfViewerComponent: PdfViewerComponent;

  constructor(private readonly componentFactoryResolver: ComponentFactoryResolver,
              private readonly _ngZone: NgZone,
              private readonly _domSanitizer: DomSanitizer) {
    super();
  }

  canHandle(fileInfo: FileInput): boolean {
    if (!isTruthyObject(fileInfo)) {
      return false;
    } else {
      const canHandleByExtension = this.canHandleByExtension(fileInfo);
      const canHandleByMimeTypeContentType = this.canHandleByMimeTypeContentType(fileInfo);
      const canHandleByPuid = this.canHandleByPuid(fileInfo);
      return canHandleByExtension || canHandleByMimeTypeContentType || canHandleByPuid;
    }
  }

  canHandleByExtension(fileInfo: FileInput): boolean {
    if (!isTruthyObject(fileInfo)) {
      return false;
    } else {
      return environment.visualizationPdfExtensions.some(value => value === fileInfo.fileExtension);
    }
  }

  canHandleWithSize(fileInfo: FileInput): boolean {
    if (!isTruthyObject(fileInfo)) {
      return false;
    } else {
      return fileInfo.dataFile.fileSize <= environment.visualizationMaxSizeFile;
    }
  }

  canHandleByMimeTypeContentType(fileInfo: FileInput): boolean {
    if (!isTruthyObject(fileInfo)) {
      return false;
    } else {
      return environment.visualizationPdfContentType.some(value => value === fileInfo.dataFile.fileFormat?.contentType) || environment.visualizationPdfMimeType.some(value => value === fileInfo.dataFile.fileFormat.contentType);
    }
  }

  isVisualizationOnGoing(fileInfo: FileInput, domElement: Element): boolean {
    return domElement.childNodes.item(0).nodeName === "PDF-VIEWER";
  }

  canHandleByPuid(fileInfo: FileInput): boolean {
    if (!isTruthyObject(fileInfo)) {
      return false;
    } else {
      return environment.visualizationPdfPronomId.some(value => value === fileInfo.dataFile.fileFormat.puid);
    }
  }

  closeVisualizer(fileInfo: FileInput, domElement: Element): void {
    domElement.remove();
  }

  openVisualizer(fileInfo: FileInput, domElement: Element, addExternalPlugin?: AddExternalComponentDirective): void {
    if (this.canHandle(fileInfo)) {
      const componentFactory = this.componentFactoryResolver.resolveComponentFactory(PdfViewerWithToolBarPresentational);
      const addExternalPluginContainerRef = addExternalPlugin.viewContainerRef;
      addExternalPluginContainerRef.clear();
      this._componentRef = addExternalPluginContainerRef.createComponent(componentFactory);
      this._toolBarPdfViewerComponent = this._componentRef.instance as PdfViewerWithToolBarPresentational;
      this._toolBarPdfViewerComponent.zoom = 1;
      this.subscribe(this._toolBarPdfViewerComponent.zoomObs,
        value => this.changeZoom(value));
      this._componentRef.changeDetectorRef.detectChanges();

      const url = URL.createObjectURL(fileInfo.blob);
      const componentFactoryNext = this.componentFactoryResolver.resolveComponentFactory(PdfViewerComponent);
      this._pdfViewerComponentRef = addExternalPluginContainerRef.createComponent(componentFactoryNext);
      this._pdfViewerComponent = this._pdfViewerComponentRef.instance as PdfViewerComponent;
      this._pdfViewerComponent.src = url;
      this._pdfViewerComponent.renderText = true;
      this._pdfViewerComponent.showBorders = true;
      this._pdfViewerComponent.ngOnInit();
      this._pdfViewerComponent.showAll = true;
      this._pdfViewerComponentRef.changeDetectorRef.detectChanges();
      const pdfElement: HTMLElement = <HTMLElement> this._pdfViewerComponentRef.location.nativeElement;
      const returnedElement = <HTMLElement> pdfElement.getElementsByClassName("ng2-pdf-viewer-container").item(0);
      returnedElement.style.maxHeight = "460px";
      returnedElement.style.overflow = "auto";
      this._pdfViewerComponentRef.changeDetectorRef.detectChanges();
    }
  }

  changeZoom(newZoom: number): void {
    this._pdfViewerComponent.zoom = newZoom;
    this._pdfViewerComponentRef.changeDetectorRef.detectChanges();
    this._pdfViewerComponent.updateSize();
    this._pdfViewerComponentRef.changeDetectorRef.detectChanges();

  }

  doAction(fileInfo: FileInput, domElement: Element): void {
  }

  isZoomSupported(): boolean {
    return false;
  }

  handleZoom(zoomEnable: boolean, fileInfo: FileInput, domElement: Element): void {
  }

  isFullScreenSupported(): boolean {
    return true;
  }
}
