import {Injectable} from "@angular/core";
import {environment} from "@environments/environment";
import {ChemicalMoleculeVisualizationEnum} from "@shared/enums/chemical-molecule-visualization.enum";
import {FileInput} from "@shared/filevisualizer/models/file-info.model";
import {AbstractFileVisualizer} from "@shared/filevisualizer/services/abstract-file-visualizer.service";
import {isTruthyObject} from "solidify-frontend";

declare var Jmol: any;

@Injectable()
export class MolFileVisualizerService extends AbstractFileVisualizer {

  type: string = "molPlugin";

  canHandle(fileInfo: FileInput): boolean {
    if (!isTruthyObject(fileInfo)) {
      return false;
    } else {
      return this.canHandleByExtension(fileInfo) || this.canHandleByMimeTypeContentType(fileInfo) || this.canHandleByPuid(fileInfo);
    }
  }

  canHandleByExtension(fileInfo: FileInput): boolean {
    if (!isTruthyObject(fileInfo)) {
      return false;
    } else {
      return environment.visualizationMolExtension.some(value => value === fileInfo.fileExtension);
    }
  }

  isVisualizationOnGoing(fileInfo: FileInput, domElement: Element): boolean {
    return false;
  }

  canHandleWithSize(fileInfo: FileInput): boolean {
    if (!isTruthyObject(fileInfo)) {
      return false;
    } else {
      return fileInfo.dataFile.fileSize <= environment.visualizationMaxSizeFile;
    }
  }

  canHandleByMimeTypeContentType(fileInfo: FileInput): boolean {
    if (!isTruthyObject(fileInfo)) {
      return false;
    } else {
      return environment.visualizationMolContentType.some(value => value === fileInfo.dataFile.fileFormat?.contentType) || environment.visualizationMolMimeType.some(value => value === fileInfo.dataFile.fileFormat.contentType);
    }
  }

  canHandleByPuid(fileInfo: FileInput): boolean {
    if (!isTruthyObject(fileInfo)) {
      return false;
    } else {
      return environment.visualizationMolPronomId.some(value => value === fileInfo.dataFile.fileFormat.puid);
    }
  }

  closeVisualizer(fileInfo: FileInput, domElement: Element): void {
    domElement.remove();
  }

  openVisualizer(fileInfo: FileInput, domElement: Element): void {
    const molUrl = URL.createObjectURL(fileInfo.blob);
    const iframe = document.createElement("iframe");
    iframe.style.overflow = "hidden";
    const envElement = JSON.stringify(environment.visualizationChemicalMoleculeMode);
    const twothreeDimension = JSON.stringify(ChemicalMoleculeVisualizationEnum.threeAndTwoDimensional);
    const threeDimension = JSON.stringify(ChemicalMoleculeVisualizationEnum.threeDimensionalOnly);
    const html = `
<button id="2d" mat-button color="primary" style="
    box-sizing: border-box;
    position: relative;
    user-select: none;
    cursor: pointer;
    outline: none;
    display: inline-block;
    white-space: nowrap;
    text-decoration: none;
    vertical-align: baseline;
    text-align: center;
    margin: 0;
    min-width: 64px;
    line-height: 36px;
    padding: 0 16px;
    border: 1px solid #0277bd;
    border-radius: 4px;
    overflow: visible;
    color: #0277bd;
    background-color: transparent;
"
onclick="Jmol.show2d(jmol, true)">2D</button>
<button id="3d" mat-button color="primary" style="
    box-sizing: border-box;
    position: relative;
    user-select: none;
    cursor: pointer;
    outline: none;
    display: inline-block;
    white-space: nowrap;
    text-decoration: none;
    vertical-align: baseline;
    text-align: center;
    margin: 0;
    min-width: 64px;
    line-height: 36px;
    padding: 0 16px;
    border: 1px solid #0277bd;
    border-radius: 4px;
    overflow: visible;
    color: #0277bd;
    background-color: transparent;
" onclick="Jmol.show2d(jmol, false)">3D</button>
<script src="./assets/jsmol/JSmol.min.js" type="text/javascript"></script>
<script src="./assets/jsmol/js/JSmolJME.js" type="text/javascript"></script>
<script src="./assets/jsmol/jsme/jsme/jsme.nocache.js" type="text/javascript"></script>
<script>
var jmolInfo = {
  color: "#FFFFFF",
  height: 375,
  width: 450,
  script: "set antialiasDisplay true;load ${molUrl} ;cartoon on;color cartoon structure;rotate z 118.48; rotate y 117.66; rotate z -47.64;;",
  use: "HTML5",
  j2sPath: "./assets/jsmol/j2s",
};
var jmeInfo = {
  use: "HTML5",
  height: 300,
  width: 450,
  visible: true,
  options : "depict"
};
if(${envElement} === ${twothreeDimension} || ${envElement} === ${threeDimension}){
  var jmol = Jmol.getApplet("jmol", jmolInfo);
if(${envElement} === ${twothreeDimension}){
    jmol._readyFunction = () => setTimeout(() => jme._loadFromJmol(jmol));
var jme = Jmol.getJMEApplet("jme", jmeInfo, jmol);
Jmol.show2d(jmol, true);
}
}

</script>
`;
    if (environment.visualizationChemicalMoleculeMode === ChemicalMoleculeVisualizationEnum.disabled) {
      this.closeVisualizer(fileInfo, domElement);
    } else {
      domElement.appendChild(iframe);
      iframe.id = "visualizationMolIfrane";
      iframe.contentWindow.document.open();
      iframe.contentWindow.document.write(html);
      iframe.contentWindow.document.close();
      iframe.width = "500px";
      iframe.height = "440px";
      iframe.style.border = "0";
      const x = document.getElementById("visualizationMolIfrane") as any;
      const button2d = x.contentDocument.getElementById("2d");
      const button3d = x.contentDocument.getElementById("3d");
      if (environment.visualizationChemicalMoleculeMode === ChemicalMoleculeVisualizationEnum.threeDimensionalOnly) {
        button2d.style.display = "none";
        button3d.style.display = "none";
      }
      let y = (x.contentWindow || x.contentDocument);
      if (y.document) {

        y = y.document;

      }
      y.body.style.overflow = "hidden";
      y.body.style.height = "404 !important";
    }
  }

  doAction(fileInfo: FileInput, domElement: Element): void {

  }

  isZoomSupported(): boolean {
    return false;
  }

  handleZoom(zoomEnable: boolean, fileInfo: FileInput, domElement: Element): void {
  }

  isFullScreenSupported(): boolean {
    return true;
  }
}
