import {Injectable} from "@angular/core";
import {environment} from "@environments/environment";
import {AddExternalComponentDirective} from "@shared/filevisualizer/directives/add-external-component.directive";
import {FileInput} from "@shared/filevisualizer/models/file-info.model";
import {AbstractFileVisualizer} from "@shared/filevisualizer/services/abstract-file-visualizer.service";
import {isTruthyObject} from "solidify-frontend";

@Injectable()
export class ImageFileVisualizerService extends AbstractFileVisualizer {

  type: string = "imagePlugin";

  img: HTMLImageElement = null;

  canHandle(fileInfo: FileInput): boolean {
    if (!isTruthyObject(fileInfo)) {
      return false;
    } else {
      const canHandleByExtension = this.canHandleByExtension(fileInfo);
      const canHandleByMimeTypeContentType = this.canHandleByMimeTypeContentType(fileInfo);
      const canHandleByPuid = this.canHandleByPuid(fileInfo);
      return canHandleByExtension || canHandleByMimeTypeContentType || canHandleByPuid;
    }
  }

  canHandleByExtension(fileInfo: FileInput): boolean {
    if (!isTruthyObject(fileInfo)) {
      return false;
    } else {
      return environment.visualizationImageExtensions.some(value => value === fileInfo.fileExtension);
    }
  }

  isVisualizationOnGoing(fileInfo: FileInput, domElement: Element): boolean {
    return isTruthyObject(this.img);
  }

  canHandleWithSize(fileInfo: FileInput): boolean {
    if (!isTruthyObject(fileInfo)) {
      return false;
    } else {
      return fileInfo.dataFile.fileSize <= environment.visualizationMaxSizeFile;
    }
  }

  canHandleByMimeTypeContentType(fileInfo: FileInput): boolean {
    if (!isTruthyObject(fileInfo)) {
      return false;
    } else {
      return environment.visualizationImageContentType.some(value => value === fileInfo?.dataFile?.fileFormat?.contentType) || environment.visualizationImageMimeType.some(value => value === fileInfo.dataFile.fileFormat.contentType);
    }
  }

  canHandleByPuid(fileInfo: FileInput): boolean {
    if (!isTruthyObject(fileInfo)) {
      return false;
    } else {
      return environment.visualizationImagePronomId.some(value => value === fileInfo.dataFile.fileFormat.puid);
    }
  }

  closeVisualizer(fileInfo: FileInput, domElement: Element): void {
    domElement.remove();
    this.img = null;
  }

  openVisualizer(fileInfo: FileInput, domElement: Element, addExternalPlugin?: AddExternalComponentDirective): void {
    this.img = document.createElement("img");
    const blob = new Blob([fileInfo.blob], {type: fileInfo.blob.type === "image/svg" ? "image/svg+xml" : fileInfo.blob.type});
    const url = URL.createObjectURL(blob);
    this.img.src = url;
    this.img.style.maxHeight = "100%";
    this.img.style.maxWidth = "100%";
    this.img.style.objectFit = "contain";
    domElement.appendChild(this.img);
  }

  doAction(fileInfo: FileInput, domElement: Element): void {
    if (domElement.classList.contains("zoom-in")) {
      domElement.removeChild(this.img);
      domElement.classList.remove("zoom-in");
      domElement.classList.add("zoom-out");
      this.img.style.height = "1000px";
      this.img.style.width = "1000px";
      domElement.appendChild(this.img);
      return;
    }
    if (domElement.classList.contains("zoom-out")) {
      domElement.removeChild(this.img);
      domElement.classList.remove("zoom-out");
      domElement.classList.add("zoom-in");
      this.img.style.height = "500px";
      this.img.style.width = "500px";
      domElement.appendChild(this.img);
      return;
    }

  }

  isZoomSupported(): boolean {
    return true;
  }

  handleZoom(zoomEnable: boolean, fileInfo: FileInput, domElement: Element): void {
    if (zoomEnable) {
      domElement.classList.add("zoom-in");
    } else {
      //remove zooms properties and resize image
      if (domElement.classList.contains("zoom-in") || domElement.classList.contains("zoom-out")) {
        domElement.removeChild(this.img);
        if (domElement.classList.contains("zoom-out")) {
          domElement.classList.remove("zoom-out");
        } else if (domElement.classList.contains("zoom-in")) {
          domElement.classList.remove("zoom-in");
        }
        this.img.style.maxHeight = "100%";
        this.img.style.maxWidth = "100%";
        this.img.style.width = null;
        this.img.style.height = null;
        domElement.appendChild(this.img);
      }
    }
  }

  isFullScreenSupported(): boolean {
    return true;
  }
}
