import {
  AfterContentInit,
  Component,
  ElementRef,
  HostListener,
  Inject,
  Input,
  OnDestroy,
  Output,
  ViewChild,
} from "@angular/core";
import {Store} from "@ngxs/store";
import {AddExternalComponentDirective} from "@shared/filevisualizer/directives/add-external-component.directive";
import {FileVisualizerHelper} from "@shared/filevisualizer/helpers/file-visualizer.helper";
import {BlobElementModel} from "@shared/filevisualizer/models/blob-element.model";
import {FileInput} from "@shared/filevisualizer/models/file-info.model";
import {
  AbstractFileVisualizer,
  FILE_VISUALIZERS,
} from "@shared/filevisualizer/services/abstract-file-visualizer.service";
import {DownloadService} from "@shared/services/download.service";
import {SharedVisualizationState} from "@shared/stores/visualization/shared-visualization-state.service";
import {SharedVisualizationAction} from "@shared/stores/visualization/shared-visualization.action";
import {
  BehaviorSubject,
  Observable,
} from "rxjs";
import {
  filter,
  take,
} from "rxjs/operators";
import {
  CoreAbstractAngularElement,
  isNotNullNorUndefined,
  isTruthyObject,
  MARK_AS_TRANSLATABLE,
  MemoizedUtil,
  NotificationService,
  ObservableUtil,
} from "solidify-frontend";

@Component({
  selector: "dlcm-file-visualizer",
  templateUrl: "./file-visualizer.component.html",
  styleUrls: ["./file-visualizer.component.scss"],
})
export class FileVisualizerPresentational extends CoreAbstractAngularElement implements AfterContentInit, OnDestroy {

  isLoadingVisualizationObs: Observable<boolean> = MemoizedUtil.isLoading(this._store, SharedVisualizationState);

  previewFile: Observable<BlobElementModel> = MemoizedUtil.select(this._store, SharedVisualizationState, state =>
    ({
      blob: state.blob,
      blobId: state.blobId,
    }));

  private static filePreviewId: number = -1;

  @HostListener("click", ["$event"])
  click(mouseEvent: MouseEvent): void {
    mouseEvent.stopPropagation();
    const elem = mouseEvent.target as Element;
    if (this.visualizationContainer.nativeElement.contains(elem)) {
      const element = this._fileVisualizers.find(value => value.isVisualizationOnGoing(this.fileToVisualize, this.visualizationContainer.nativeElement));
      element?.doAction(this.fileToVisualize, this.visualizationContainer.nativeElement);
    }
  }

  @ViewChild("visualizationContainer", {static: true})
  visualizationContainer: ElementRef<Element>;

  @ViewChild(AddExternalComponentDirective, {static: true}) addExternalPlugin: AddExternalComponentDirective;

  private readonly _closeBS: BehaviorSubject<boolean> = new BehaviorSubject<boolean>(false);

  @Output("closeVisualizerChange")
  readonly closeVisualizerObs: Observable<boolean> = ObservableUtil.asObservable(this._closeBS);

  private readonly _errorMessageBS: BehaviorSubject<string> = new BehaviorSubject<string>("");

  @Output("errorMessageChange")
  readonly errorMessageObs: Observable<string> = ObservableUtil.asObservable(this._errorMessageBS);

  private readonly _fullScreenAvailableBS: BehaviorSubject<boolean | undefined> = new BehaviorSubject<boolean>(undefined);

  @Output("fullScreenAvailableChange")
  readonly fullScreenAvailableObs: Observable<boolean | undefined> = ObservableUtil.asObservable(this._fullScreenAvailableBS);

  @Input()
  visualizationContainerInput: ElementRef<Element>;

  @Input()
  fileToVisualize: FileInput;

  _zoomEnable: boolean = false;

  @Input()
  set zoomEnable(zoomEnable: boolean) {
    if (zoomEnable !== this._zoomEnable) {
      this.updateZoom();
    }
    this._zoomEnable = zoomEnable;
  }

  constructor(@Inject(FILE_VISUALIZERS) private readonly _fileVisualizers: AbstractFileVisualizer[],
              private readonly _store: Store,
              private _notificationService: NotificationService,
              private readonly _downloadService: DownloadService) {
    super();
  }

  ngAfterContentInit(): void {
    if (isTruthyObject(this.fileToVisualize)) {
      this.visualize(this.fileToVisualize, this.visualizationContainer.nativeElement);
    }
  }

  ngOnDestroy(): void {
    super.ngOnDestroy();
    this.closeVisualization();
  }

  canHandleBySize(fileInfo: FileInput): boolean {
    return isTruthyObject(fileInfo) && this._fileVisualizers.some(value => value.canHandleWithSize(fileInfo));
  }

  visualize(fileInfoParam: FileInput, documentElement: Element): void {
    const fileExtention = FileVisualizerHelper.getFileExtension(fileInfoParam.dataFile.fileName);
    let fileInfo: FileInput = {
      fileExtension: fileExtention,
      dataFile: fileInfoParam.dataFile,
      blob: fileInfoParam?.blob,
      container: documentElement,
    };
    if (!FileVisualizerHelper.canHandle(this._fileVisualizers, fileInfo)) {
      this.closeVisualization();
      if (!this.canHandleBySize(fileInfo)) {
        const sizeNotSupported = MARK_AS_TRANSLATABLE("notification.file.visualizer.file.tooBig");
        this._notificationService.showInformation(sizeNotSupported);
        this._errorMessageBS.next(sizeNotSupported);
      } else {
        const notSupportedMessage = MARK_AS_TRANSLATABLE("notification.file.visualizer.file.notSupported");
        this._notificationService.showInformation(notSupportedMessage);
        this._errorMessageBS.next(notSupportedMessage);
      }
      return;
    }
    const filePreviewId = FileVisualizerPresentational.filePreviewId++;
    const handleByPronomId = this._fileVisualizers.find(value => value.canHandleByPuid(fileInfo));
    const handleByContentType = this._fileVisualizers.find(value => value.canHandleByMimeTypeContentType(fileInfo));
    const handleByExtension = this._fileVisualizers.find(value => value.canHandleByExtension(fileInfo));
    this.subscribe(this.previewFile.pipe(
      filter(value => isTruthyObject(value.blob) && filePreviewId === value.blobId),
      take(1)),
      value => {
        fileInfo = {
          fileExtension: fileExtention,
          dataFile: fileInfo.dataFile,
          container: documentElement,
          blob: value.blob,
        };
        if (isTruthyObject(handleByPronomId)) {
          handleByPronomId.openVisualizer(fileInfo, documentElement, this.addExternalPlugin);
          this._fullScreenAvailableBS.next(handleByPronomId.isFullScreenSupported());
        } else if (isTruthyObject(handleByContentType)) {
          if (this.isPlainTextFile(fileInfo) && !isTruthyObject(handleByExtension)) {
            this._fileVisualizers.find(visualizer => visualizer.type === "textPlugin").openVisualizer(fileInfo, documentElement);
          } else if (!isTruthyObject(handleByExtension)) {
            handleByContentType.openVisualizer(fileInfo, documentElement, this.addExternalPlugin);
            this._fullScreenAvailableBS.next(handleByContentType.isFullScreenSupported());
          } else {
            handleByExtension.openVisualizer(fileInfo, documentElement, this.addExternalPlugin);
            this._fullScreenAvailableBS.next(handleByExtension.isFullScreenSupported());
          }
        } else if (isTruthyObject(handleByExtension)) {
          handleByExtension.openVisualizer(fileInfo, documentElement, this.addExternalPlugin);
          this._fullScreenAvailableBS.next(handleByExtension.isFullScreenSupported());
        }
      });
    this._store.dispatch(new SharedVisualizationAction.Download(fileInfo, filePreviewId));
  }

  onClose(needToClose: MouseEvent): void {
    this.closeVisualization();
  }

  isPlainTextFile(fileInfo: FileInput): boolean {
    return fileInfo?.dataFile?.fileFormat?.contentType === "text/plain" || fileInfo?.dataFile?.fileFormat?.contentType === "text/xml";
  }

  closeVisualization(): void {
    const fileVizualizerOnGoing = this._fileVisualizers.find(value =>
      value.isVisualizationOnGoing(this.fileToVisualize, this.visualizationContainer.nativeElement));
    if (isTruthyObject(fileVizualizerOnGoing)) {
      fileVizualizerOnGoing.closeVisualizer(this.fileToVisualize, this.visualizationContainer.nativeElement);
    }
    this._closeBS.next(true);
  }

  updateZoom(): void {
    const visualizerSupported = this._fileVisualizers.find(value => value.isZoomSupported());
    if (isNotNullNorUndefined(visualizerSupported)) {
      visualizerSupported.handleZoom(!this._zoomEnable, this.fileToVisualize, this.visualizationContainer.nativeElement);
    }
  }
}
