import {HttpParams} from "@angular/common/http";
import {Injectable} from "@angular/core";
import {
  ActivatedRoute,
  Router,
} from "@angular/router";
import {Enums} from "@enums";
import {Aip} from "@models";
import {Navigate} from "@ngxs/router-plugin";
import {
  Action,
  Actions,
  Selector,
  State,
  StateContext,
  Store,
} from "@ngxs/store";

import {ApiActionEnum} from "@shared/enums/api-action.enum";
import {LocalStateEnum} from "@shared/enums/local-state.enum";
import {
  AppRoutesEnum,
  RoutesEnum,
  SharedAipRoutesEnum,
  urlSeparator,
} from "@shared/enums/routes.enum";
import {AipHelper} from "@shared/features/aip/helpers/aip.helper";
import {
  defaultSharedAipCollectionValue,
  SharedAipCollectionState,
  SharedAipCollectionStateModel,
} from "@shared/features/aip/stores/aip-collection/shared-aip-collection.state";
import {
  defaultSharedAipDataFileValue,
  SharedAipDataFileState,
  SharedAipDataFileStateModel,
} from "@shared/features/aip/stores/aip-data-file/shared-aip-data-file.state";
import {
  defaultSharedAipOrgUnitValue,
  SharedAipOrganizationalUnitState,
  SharedAipOrganizationalUnitStateModel,
} from "@shared/features/aip/stores/organizational-unit/shared-aip-organizational-unit.state";
import {
  SharedAipAction,
  sharedAipActionNameSpace,
} from "@shared/features/aip/stores/shared-aip.action";
import {
  SharedAipStatusHistoryState,
  SharedAipStatusHistoryStateModel,
} from "@shared/features/aip/stores/status-history/shared-aip-status-history.state";
import {Result} from "@shared/models/business/result.model";
import {DownloadService} from "@shared/services/download.service";
import {defaultStatusHistoryInitValue} from "@shared/stores/status-history/status-history.state";
import {Observable} from "rxjs";
import {
  catchError,
  tap,
} from "rxjs/operators";
import {
  ApiService,
  defaultResourceStateInitValue,
  isNullOrUndefined,
  MARK_AS_TRANSLATABLE,
  NotificationService,
  OverrideDefaultAction,
  ResourceState,
  ResourceStateModel,
  SolidifyStateError,
  StoreUtil,
} from "solidify-frontend";

export interface SharedAipStateModel extends ResourceStateModel<Aip> {
  [LocalStateEnum.shared_aip_dataFile]: SharedAipDataFileStateModel;
  [LocalStateEnum.shared_aip_organizationalUnit]: SharedAipOrganizationalUnitStateModel;
  [LocalStateEnum.shared_aip_collection]: SharedAipCollectionStateModel;
  preservationPlanning_aip_statusHistory: SharedAipStatusHistoryStateModel;
}

@Injectable()
@State<SharedAipStateModel>({
  name: LocalStateEnum.shared_aip,
  defaults: {
    ...defaultResourceStateInitValue(),
    [LocalStateEnum.shared_aip_dataFile]: defaultSharedAipDataFileValue(),
    [LocalStateEnum.shared_aip_organizationalUnit]: defaultSharedAipOrgUnitValue(),
    [LocalStateEnum.shared_aip_collection]: defaultSharedAipCollectionValue(),
    preservationPlanning_aip_statusHistory: {...defaultStatusHistoryInitValue()},
  },
  children: [
    SharedAipOrganizationalUnitState,
    SharedAipCollectionState,
    SharedAipStatusHistoryState,
    SharedAipDataFileState,
  ],
})
export class SharedAipState extends ResourceState<SharedAipStateModel, Aip> {
  private readonly KEY_QUERY_PARAM_REASON: string = "reason";
  private readonly KEY_QUERY_PARAM_DURATION: string = "duration";

  constructor(protected apiService: ApiService,
              protected store: Store,
              protected notificationService: NotificationService,
              protected actions$: Actions,
              private readonly _router: Router,
              private readonly _route: ActivatedRoute,
              private downloadService: DownloadService) {
    super(apiService, store, notificationService, actions$, {
      nameSpace: sharedAipActionNameSpace,
      routeRedirectUrlAfterSuccessDeleteAction: RoutesEnum.preservationPlanningAipDownloaded,
      notificationResourceDeleteSuccessTextToTranslate: MARK_AS_TRANSLATABLE("aipDownloaded.notification.resource.delete.success"),
      notificationResourceDeleteFailTextToTranslate: MARK_AS_TRANSLATABLE("aipDownloaded.notification.resource.delete.fail"),
    });
  }

  protected get _urlResource(): string {
    return AipHelper.generateUrlResource(this.store);
  }

  @Selector()
  static currentAipName(state: SharedAipStateModel): string | undefined {
    return state.current.info.name;
  }

  @Selector()
  static isLoadingWithDependency(state: SharedAipStateModel): boolean {
    return this.isLoading(state);
  }

  @Selector()
  static isReadyToBeDisplayed(state: SharedAipStateModel): boolean {
    return !isNullOrUndefined(state.current);
  }

  @Selector()
  static isLoading(state: SharedAipStateModel): boolean {
    return StoreUtil.isLoadingState(state);
  }

  @Action(SharedAipAction.Download)
  download(ctx: StateContext<SharedAipStateModel>, action: SharedAipAction.Download): void {
    const fileName = "aip_" + action.id + ".zip";
    this.downloadService.download(false, `${this._urlResource}/${action.id}/${ApiActionEnum.DL}`, fileName);
  }

  @OverrideDefaultAction()
  @Action(SharedAipAction.GetById)
  getById(ctx: StateContext<SharedAipStateModel>, action: SharedAipAction.GetById): Observable<Aip> {
    let reset = {};
    if (!action.keepCurrentContext) {
      reset = {
        current: undefined,
      };
    }
    ctx.patchState({
      isLoadingCounter: ctx.getState().isLoadingCounter + 1,
      ...reset,
    });

    const url = AipHelper.generateUrlResource(this.store, action.storagionNumber, action.url);
    return this.apiService.getById<Aip>(url, action.id)
      .pipe(
        tap((model: Aip) => {
          ctx.dispatch(new SharedAipAction.GetByIdSuccess(action, model));
        }),
        catchError(error => {
          ctx.dispatch(new SharedAipAction.GetByIdFail(action));
          throw new SolidifyStateError(this, error);
        }),
      );
  }

  @Action(SharedAipAction.Reindex)
  reindex(ctx: StateContext<SharedAipStateModel>, action: SharedAipAction.Reindex): Observable<Result> {
    ctx.patchState({
      isLoadingCounter: ctx.getState().isLoadingCounter + 1,
    });
    const params = new HttpParams().set("level", "deep");
    return this.apiService.post<any, Result>(`${this._urlResource}/${action.id}/${ApiActionEnum.REINDEX}`, {params: params})
      .pipe(
        tap(result => {
          if (result?.status === Enums.Result.ActionStatusEnum.EXECUTED) {
            ctx.dispatch(new SharedAipAction.ReindexSuccess(action));
          } else {
            ctx.dispatch(new SharedAipAction.ReindexFail(action));
          }
        }),
        catchError(error => {
          ctx.dispatch(new SharedAipAction.ReindexFail(action));
          throw error;
        }),
      );
  }

  @Action(SharedAipAction.ReindexSuccess)
  reindexSuccess(ctx: StateContext<SharedAipStateModel>, action: SharedAipAction.ReindexSuccess): void {
    ctx.patchState({
      isLoadingCounter: ctx.getState().isLoadingCounter - 1,
    });
    this.notificationService.showInformation(MARK_AS_TRANSLATABLE("notification.aip.action.reindex.success"));
  }

  @Action(SharedAipAction.ReindexFail)
  reindexFail(ctx: StateContext<SharedAipStateModel>, action: SharedAipAction.ReindexFail): void {
    ctx.patchState({
      isLoadingCounter: ctx.getState().isLoadingCounter - 1,
    });
    this.notificationService.showError(MARK_AS_TRANSLATABLE("notification.aip.action.reindex.fail"));
  }

  @Action(SharedAipAction.DeepChecksum)
  deepChecksum(ctx: StateContext<SharedAipStateModel>, action: SharedAipAction.DeepChecksum): Observable<Result> {
    ctx.patchState({
      isLoadingCounter: ctx.getState().isLoadingCounter + 1,
    });
    const params = new HttpParams().set("level", "deep");
    return this.apiService.post<any, Result>(`${this._urlResource}/${action.id}/${ApiActionEnum.CHECK_FIXITY}`, {params: params})
      .pipe(
        tap(result => {
          if (result?.status === Enums.Result.ActionStatusEnum.EXECUTED) {
            ctx.dispatch(new SharedAipAction.DeepChecksumSuccess(action));
          } else {
            ctx.dispatch(new SharedAipAction.DeepChecksumFail(action));
          }
        }),
        catchError(error => {
          ctx.dispatch(new SharedAipAction.DeepChecksumFail(action));
          throw error;
        }),
      );
  }

  @Action(SharedAipAction.DeepChecksumSuccess)
  deepChecksumSuccess(ctx: StateContext<SharedAipStateModel>, action: SharedAipAction.DeepChecksumSuccess): void {
    ctx.patchState({
      isLoadingCounter: ctx.getState().isLoadingCounter - 1,
    });
    this.notificationService.showInformation(MARK_AS_TRANSLATABLE("notification.aip.action.deepChecksum.success"));
  }

  @Action(SharedAipAction.DeepChecksumFail)
  deepChecksumFail(ctx: StateContext<SharedAipStateModel>, action: SharedAipAction.DeepChecksumFail): void {
    ctx.patchState({
      isLoadingCounter: ctx.getState().isLoadingCounter - 1,
    });
    this.notificationService.showError(MARK_AS_TRANSLATABLE("notification.aip.action.deepChecksum.fail"));
  }

  @Action(SharedAipAction.SimpleChecksum)
  simpleChecksum(ctx: StateContext<SharedAipStateModel>, action: SharedAipAction.SimpleChecksum): Observable<Result> {
    ctx.patchState({
      isLoadingCounter: ctx.getState().isLoadingCounter + 1,
    });
    const params = new HttpParams().set("level", "simple");
    return this.apiService.post<any, Result>(`${this._urlResource}/${action.id}/${ApiActionEnum.CHECK_FIXITY}`, {params: params})
      .pipe(
        tap(result => {
          if (result?.status === Enums.Result.ActionStatusEnum.EXECUTED) {
            ctx.dispatch(new SharedAipAction.SimpleChecksumSuccess(action));
          } else {
            ctx.dispatch(new SharedAipAction.SimpleChecksumFail(action));
          }
        }),
        catchError(error => {
          ctx.dispatch(new SharedAipAction.SimpleChecksumFail(action));
          throw error;
        }),
      );
  }

  @Action(SharedAipAction.SimpleChecksumSuccess)
  simpleChecksumSuccess(ctx: StateContext<SharedAipStateModel>, action: SharedAipAction.SimpleChecksumSuccess): void {
    ctx.patchState({
      isLoadingCounter: ctx.getState().isLoadingCounter - 1,
    });
    this.notificationService.showInformation(MARK_AS_TRANSLATABLE("notification.aip.action.simpleChecksum.success"));
  }

  @Action(SharedAipAction.SimpleChecksumFail)
  simpleChecksumFail(ctx: StateContext<SharedAipStateModel>, action: SharedAipAction.SimpleChecksumFail): void {
    ctx.patchState({
      isLoadingCounter: ctx.getState().isLoadingCounter - 1,
    });
    this.notificationService.showError(MARK_AS_TRANSLATABLE("notification.aip.action.simpleChecksum.fail"));
  }

  @Action(SharedAipAction.Resume)
  resume(ctx: StateContext<SharedAipStateModel>, action: SharedAipAction.Resume): Observable<Result> {
    ctx.patchState({
      isLoadingCounter: ctx.getState().isLoadingCounter + 1,
    });
    return this.apiService.post<any, Result>(`${this._urlResource}/${action.id}/${ApiActionEnum.RESUME}`, null)
      .pipe(
        tap(result => {
          if (result?.status === Enums.Result.ActionStatusEnum.EXECUTED) {
            ctx.dispatch(new SharedAipAction.ResumeSuccess(action));
          } else {
            ctx.dispatch(new SharedAipAction.ResumeFail(action));
          }
        }),
        catchError(error => {
          ctx.dispatch(new SharedAipAction.ResumeFail(action));
          throw error;
        }),
      );
  }

  @Action(SharedAipAction.ResumeSuccess)
  resumeSuccess(ctx: StateContext<SharedAipStateModel>, action: SharedAipAction.ResumeSuccess): void {
    ctx.patchState({
      isLoadingCounter: ctx.getState().isLoadingCounter - 1,
    });
    this.notificationService.showInformation(MARK_AS_TRANSLATABLE("notification.aip.action.resume.success"));
  }

  @Action(SharedAipAction.ResumeFail)
  resumeFail(ctx: StateContext<SharedAipStateModel>, action: SharedAipAction.ResumeFail): void {
    ctx.patchState({
      isLoadingCounter: ctx.getState().isLoadingCounter - 1,
    });
    this.notificationService.showError(MARK_AS_TRANSLATABLE("notification.aip.action.resume.fail"));
  }

  @Action(SharedAipAction.GoToAip)
  goToAip(ctx: StateContext<SharedAipStateModel>, action: SharedAipAction.GoToAip): void {
    ctx.patchState({
      current: action.aip,
    });
    const pathAipDetail = RoutesEnum.preservationPlanningAip + urlSeparator + action.storagion_number + urlSeparator + SharedAipRoutesEnum.aipDetail + AppRoutesEnum.separator;
    const path = [pathAipDetail, action.aip.resId];
    ctx.dispatch(new Navigate(path));
  }

  @Action(SharedAipAction.Check)
  check(ctx: StateContext<SharedAipStateModel>, action: SharedAipAction.Check): Observable<Result> {
    ctx.patchState({
      isLoadingCounter: ctx.getState().isLoadingCounter + 1,
    });
    return this.apiService.post<any, Result>(`${this._urlResource}/${action.id}/${ApiActionEnum.CHECK}`, null)
      .pipe(
        tap(result => {
          if (result?.status === Enums.Result.ActionStatusEnum.EXECUTED) {
            ctx.dispatch(new SharedAipAction.CheckSuccess(action));
          } else {
            ctx.dispatch(new SharedAipAction.CheckFail(action));
          }
        }),
        catchError(error => {
          ctx.dispatch(new SharedAipAction.CheckFail(action));
          throw error;
        }),
      );
  }

  @Action(SharedAipAction.CheckSuccess)
  checkSuccess(ctx: StateContext<SharedAipStateModel>, action: SharedAipAction.CheckSuccess): void {
    ctx.patchState({
      isLoadingCounter: ctx.getState().isLoadingCounter - 1,
    });
    this.notificationService.showInformation(MARK_AS_TRANSLATABLE("notification.aip.action.check.success"));
  }

  @Action(SharedAipAction.CheckFail)
  checkFail(ctx: StateContext<SharedAipStateModel>, action: SharedAipAction.CheckFail): void {
    ctx.patchState({
      isLoadingCounter: ctx.getState().isLoadingCounter - 1,
    });
    this.notificationService.showError(MARK_AS_TRANSLATABLE("notification.aip.action.check.fail"));
  }

  @Action(SharedAipAction.Dispose)
  dispose(ctx: StateContext<SharedAipStateModel>, action: SharedAipAction.Dispose): Observable<string> {
    ctx.patchState({
      isLoadingCounter: ctx.getState().isLoadingCounter + 1,
    });
    return this.apiService.delete<string>(`${this._urlResource}/${action.id}`, null)
      .pipe(
        tap(result => {
          ctx.dispatch(new SharedAipAction.DisposeSuccess(action));
        }),
        catchError(error => {
          ctx.dispatch(new SharedAipAction.DisposeFail(action));
          throw error;
        }),
      );
  }

  @Action(SharedAipAction.DisposeSuccess)
  disposeSuccess(ctx: StateContext<SharedAipStateModel>, action: SharedAipAction.DisposeSuccess): void {
    ctx.patchState({
      isLoadingCounter: ctx.getState().isLoadingCounter - 1,
    });
    this.notificationService.showInformation(MARK_AS_TRANSLATABLE("notification.aip.action.dispose.success"));
  }

  @Action(SharedAipAction.DisposeFail)
  disposeFail(ctx: StateContext<SharedAipStateModel>, action: SharedAipAction.DisposeFail): void {
    ctx.patchState({
      isLoadingCounter: ctx.getState().isLoadingCounter - 1,
    });
    this.notificationService.showError(MARK_AS_TRANSLATABLE("notification.aip.action.dispose.fail"));
  }

  @Action(SharedAipAction.ApproveDisposal)
  approveDisposal(ctx: StateContext<SharedAipStateModel>, action: SharedAipAction.ApproveDisposal): Observable<Result> {
    ctx.patchState({
      isLoadingCounter: ctx.getState().isLoadingCounter + 1,
    });
    const customParams = {
      [this.KEY_QUERY_PARAM_REASON]: action.reason,
    };
    return this.apiService.post<any, Result>(`${this._urlResource}/${action.id}/${ApiActionEnum.APPROVE_DISPOSAL}`, null, customParams)
      .pipe(
        tap(result => {
          if (result?.status === Enums.Result.ActionStatusEnum.EXECUTED) {
            ctx.dispatch(new SharedAipAction.ApproveDisposalSuccess(action));
          } else {
            ctx.dispatch(new SharedAipAction.ApproveDisposalFail(action));
          }
        }),
        catchError(error => {
          ctx.dispatch(new SharedAipAction.ApproveDisposalFail(action));
          throw error;
        }),
      );
  }

  @Action(SharedAipAction.ApproveDisposalSuccess)
  approveDisposalSuccess(ctx: StateContext<SharedAipStateModel>, action: SharedAipAction.ApproveDisposalSuccess): void {
    ctx.patchState({
      isLoadingCounter: ctx.getState().isLoadingCounter - 1,
    });
    this.store.dispatch(new SharedAipAction.GetById(ctx.getState().current.resId));
    this.notificationService.showInformation(MARK_AS_TRANSLATABLE("notification.aip.action.approveDisposal.success"));
  }

  @Action(SharedAipAction.ApproveDisposalFail)
  approveDisposalFail(ctx: StateContext<SharedAipStateModel>, action: SharedAipAction.ApproveDisposalFail): void {
    ctx.patchState({
      isLoadingCounter: ctx.getState().isLoadingCounter - 1,
    });
    this.notificationService.showError(MARK_AS_TRANSLATABLE("notification.aip.action.approveDisposal.fail"));
  }

  @Action(SharedAipAction.ApproveDisposalByOrgunit)
  approveDisposalByOrgunit(ctx: StateContext<SharedAipStateModel>, action: SharedAipAction.ApproveDisposalByOrgunit): Observable<Result> {
    ctx.patchState({
      isLoadingCounter: ctx.getState().isLoadingCounter + 1,
    });
    return this.apiService.post<any, Result>(`${this._urlResource}/${action.id}/${ApiActionEnum.APPROVE_DISPOSAL_BY_ORGUNIT}`, null, {
      [this.KEY_QUERY_PARAM_REASON]: action.reason,
    })
      .pipe(
        tap(result => {
          if (result?.status === Enums.Result.ActionStatusEnum.EXECUTED) {
            ctx.dispatch(new SharedAipAction.ApproveDisposalByOrgunitSuccess(action));
          } else {
            ctx.dispatch(new SharedAipAction.ApproveDisposalByOrgunitFail(action));
          }
        }),
        catchError(error => {
          ctx.dispatch(new SharedAipAction.ApproveDisposalByOrgunitFail(action));
          throw error;
        }),
      );
  }

  @Action(SharedAipAction.ApproveDisposalByOrgunitSuccess)
  approveDisposalByOrgunitSuccess(ctx: StateContext<SharedAipStateModel>, action: SharedAipAction.ApproveDisposalByOrgunitSuccess): void {
    ctx.patchState({
      isLoadingCounter: ctx.getState().isLoadingCounter - 1,
    });
    this.store.dispatch(new SharedAipAction.GetById(ctx.getState().current.resId));
    this.notificationService.showInformation(MARK_AS_TRANSLATABLE("notification.aip.action.approveDisposalByOrgunit.success"));
  }

  @Action(SharedAipAction.ApproveDisposalByOrgunitFail)
  approveDisposalByOrgunitFail(ctx: StateContext<SharedAipStateModel>, action: SharedAipAction.ApproveDisposalByOrgunitFail): void {
    ctx.patchState({
      isLoadingCounter: ctx.getState().isLoadingCounter - 1,
    });
    this.notificationService.showError(MARK_AS_TRANSLATABLE("notification.aip.action.approveDisposalByOrgunit.fail"));
  }

  @Action(SharedAipAction.ExtendRetention)
  extendRetention(ctx: StateContext<SharedAipStateModel>, action: SharedAipAction.ExtendRetention): Observable<Result> {
    ctx.patchState({
      isLoadingCounter: ctx.getState().isLoadingCounter + 1,
    });
    return this.apiService.post<any, Result>(`${this._urlResource}/${action.id}/${ApiActionEnum.EXTEND_RETENTION}`, null, {
      [this.KEY_QUERY_PARAM_DURATION]: action.duration + "",
    })
      .pipe(
        tap(result => {
          if (result?.status === Enums.Result.ActionStatusEnum.EXECUTED) {
            ctx.dispatch(new SharedAipAction.ExtendRetentionSuccess(action));
          } else {
            ctx.dispatch(new SharedAipAction.ExtendRetentionFail(action));
          }
        }),
        catchError(error => {
          ctx.dispatch(new SharedAipAction.ExtendRetentionFail(action));
          throw error;
        }),
      );
  }

  @Action(SharedAipAction.ExtendRetentionSuccess)
  extendRetentionSuccess(ctx: StateContext<SharedAipStateModel>, action: SharedAipAction.ExtendRetentionSuccess): void {
    ctx.patchState({
      isLoadingCounter: ctx.getState().isLoadingCounter - 1,
    });
    this.store.dispatch(new SharedAipAction.GetById(ctx.getState().current.resId));
    this.notificationService.showInformation(MARK_AS_TRANSLATABLE("notification.aip.action.extendRetention.success"));
  }

  @Action(SharedAipAction.ExtendRetentionFail)
  extendRetentionFail(ctx: StateContext<SharedAipStateModel>, action: SharedAipAction.ExtendRetentionFail): void {
    ctx.patchState({
      isLoadingCounter: ctx.getState().isLoadingCounter - 1,
    });
    this.notificationService.showError(MARK_AS_TRANSLATABLE("notification.aip.action.extendRetention.fail"));
  }
}
