import {
  Inject,
  Injectable,
} from "@angular/core";
import {WINDOW} from "@app/app.module";
import {Enums} from "@enums";
import {
  Action,
  Actions,
  State,
  StateContext,
  Store,
} from "@ngxs/store";
import {SharedAipDataFileStatusHistoryStateModel} from "@preservation-planning/sip/stores/data-file/status-history/preservation-planning-sip-data-file-status-history.state";
import {ApiActionEnum} from "@shared/enums/api-action.enum";
import {ApiResourceNameEnum} from "@shared/enums/api-resource-name.enum";
import {LabelTranslateEnum} from "@shared/enums/label-translate.enum";
import {LocalStateEnum} from "@shared/enums/local-state.enum";
import {AipHelper} from "@shared/features/aip/helpers/aip.helper";
import {AipDataFile} from "@shared/features/aip/models/aip-data-file.model";
import {
  SharedAipDataFileAction,
  sharedAipDataFileActionNameSpace,
} from "@shared/features/aip/stores/aip-data-file/shared-aip-data-file.action";
import {SharedAipDataFileStatusHistoryState} from "@shared/features/aip/stores/aip-data-file/status-history/shared-aip-data-file-status-history.state";
import {Result} from "@shared/models/business/result.model";
import {DownloadService} from "@shared/services/download.service";
import {defaultStatusHistoryInitValue} from "@shared/stores/status-history/status-history.state";
import {Observable} from "rxjs";
import {
  catchError,
  tap,
} from "rxjs/operators";
import {
  ApiService,
  CompositionState,
  CompositionStateModel,
  defaultCompositionStateInitValue,
  NotificationService,
  SolidifyStateError,
} from "solidify-frontend";

export const defaultSharedAipDataFileValue: () => SharedAipDataFileStateModel = () =>
  ({
    ...defaultCompositionStateInitValue(),
    [LocalStateEnum.shared_aip_dataFile_statusHistory]: defaultStatusHistoryInitValue(),
  });

export interface SharedAipDataFileStateModel extends CompositionStateModel<AipDataFile> {
  [LocalStateEnum.shared_aip_dataFile_statusHistory]: SharedAipDataFileStatusHistoryStateModel;
}

@Injectable()
@State<SharedAipDataFileStateModel>({
  name: LocalStateEnum.shared_aip_dataFile,
  defaults: {
    ...defaultSharedAipDataFileValue(),
  },
  children: [
    SharedAipDataFileStatusHistoryState,
  ],
})
export class SharedAipDataFileState extends CompositionState<SharedAipDataFileStateModel, AipDataFile> {
  constructor(protected apiService: ApiService,
              protected store: Store,
              protected notificationService: NotificationService,
              protected actions$: Actions,
              @Inject(WINDOW) private _window: Window,
              private downloadService: DownloadService) {
    super(apiService, store, notificationService, actions$, {
      nameSpace: sharedAipDataFileActionNameSpace,
      resourceName: ApiResourceNameEnum.DATAFILE,
    });
  }

  protected get _urlResource(): string {
    return AipHelper.generateUrlResource(this.store);
  }

  @Action(SharedAipDataFileAction.Refresh)
  refresh(ctx: StateContext<SharedAipDataFileStateModel>, action: SharedAipDataFileAction.Refresh): void {
    ctx.dispatch(new SharedAipDataFileAction.GetAll(action.parentId, undefined, true));
  }

  @Action(SharedAipDataFileAction.Download)
  download(ctx: StateContext<SharedAipDataFileStateModel>, action: SharedAipDataFileAction.Download): void {
    const url = `${this._urlResource}/${action.parentId}/${this._resourceName}/${action.dataFile.resId}/${ApiActionEnum.DL}`;
    this.downloadService.download(false, url, action.dataFile.fileName, action.dataFile.fileSize);
  }

  @Action(SharedAipDataFileAction.Resume)
  resume(ctx: StateContext<SharedAipDataFileStateModel>, action: SharedAipDataFileAction.Resume): Observable<Result> {
    return this.apiService.post<any, Result>(`${this._urlResource}/${action.parentId}/${this._resourceName}/${action.dataFile.resId}/${ApiActionEnum.RESUME}`)
      .pipe(
        tap(result => {
          if (result?.status === Enums.Result.ActionStatusEnum.EXECUTED) {
            this.notificationService.showInformation(LabelTranslateEnum.resourceResumed);
            ctx.dispatch(new SharedAipDataFileAction.GetAll(action.parentId, undefined, true));
          } else {
            this.notificationService.showInformation(LabelTranslateEnum.unableResumedResource);
          }
        }),
        catchError(error => {
          this.notificationService.showInformation(LabelTranslateEnum.unableResumedResource);
          throw new SolidifyStateError(error);
        }),
      );
  }
}
