import {
  Inject,
  Injectable,
} from "@angular/core";
import {WINDOW} from "@app/app.module";
import {Aip} from "@models";
import {
  Action,
  Actions,
  State,
  StateContext,
  Store,
} from "@ngxs/store";
import {ApiActionEnum} from "@shared/enums/api-action.enum";
import {ApiResourceNameEnum} from "@shared/enums/api-resource-name.enum";
import {LocalStateEnum} from "@shared/enums/local-state.enum";
import {AipHelper} from "@shared/features/aip/helpers/aip.helper";
import {
  SharedAipCollectionAction,
  sharedAipCollectionActionNameSpace,
} from "@shared/features/aip/stores/aip-collection/shared-aip-collection.action";
import {
  SharedAipCollectionStatusHistoryState,
  SharedAipCollectionStatusHistoryStateModel,
} from "@shared/features/aip/stores/aip-collection/status-history/shared-aip-collection-status-history.state";
import {SharedAipAction} from "@shared/features/aip/stores/shared-aip.action";
import {DownloadService} from "@shared/services/download.service";
import {defaultStatusHistoryInitValue} from "@shared/stores/status-history/status-history.state";
import {
  ApiService,
  CompositionState,
  CompositionStateModel,
  defaultCompositionStateInitValue,
  NotificationService,
} from "solidify-frontend";

export const defaultSharedAipCollectionValue: () => SharedAipCollectionStateModel = () =>
  ({
    ...defaultCompositionStateInitValue(),
    [LocalStateEnum.shared_aip_collection_statusHistory]: {...defaultStatusHistoryInitValue()},
  });

export interface SharedAipCollectionStateModel extends CompositionStateModel<Aip> {
  [LocalStateEnum.shared_aip_collection_statusHistory]: SharedAipCollectionStatusHistoryStateModel;
}

@Injectable()
@State<SharedAipCollectionStateModel>({
  name: LocalStateEnum.shared_aip_collection,
  defaults: {
    ...defaultSharedAipCollectionValue(),
    [LocalStateEnum.shared_aip_collection_statusHistory]: {...defaultStatusHistoryInitValue()},
  },
  children: [
    SharedAipCollectionStatusHistoryState,
  ],
})
export class SharedAipCollectionState extends CompositionState<SharedAipCollectionStateModel, Aip> {
  constructor(protected apiService: ApiService,
              protected store: Store,
              protected notificationService: NotificationService,
              protected actions$: Actions,
              @Inject(WINDOW) private _window: Window,
              private downloadService: DownloadService) {
    super(apiService, store, notificationService, actions$, {
      nameSpace: sharedAipCollectionActionNameSpace,
      resourceName: ApiResourceNameEnum.AIP,
    });
  }

  protected get _urlResource(): string {
    return AipHelper.generateUrlResource(this.store);
  }

  @Action(SharedAipCollectionAction.Refresh)
  refresh(ctx: StateContext<SharedAipCollectionStateModel>, action: SharedAipCollectionAction.Refresh): void {
    ctx.dispatch(new SharedAipCollectionAction.GetAll(action.parentId, undefined, true));
  }

  @Action(SharedAipCollectionAction.DownloadAip)
  downloadAip(ctx: StateContext<SharedAipCollectionStateModel>, action: SharedAipCollectionAction.DownloadAip): void {
    const url = `${this._urlResource}/${action.parentId}/${this._resourceName}/${action.aip.resId}/${ApiActionEnum.DL}`;
    this.downloadService.download(false, url, action.aip.info.name, action.aip.archiveSize);
  }

  @Action(SharedAipCollectionAction.GoToAip)
  goToAip(ctx: StateContext<SharedAipCollectionStateModel>, action: SharedAipCollectionAction.GoToAip): void {
    ctx.patchState({
      current: undefined,
      list: undefined,
    });
    ctx.dispatch(new SharedAipAction.GoToAip(action.aip, action.storagion_number));
  }
}
