import {environment} from "@environments/environment";
import {Store} from "@ngxs/store";
import {ApiResourceNameEnum} from "@shared/enums/api-resource-name.enum";
import {
  AccessResourceApiEnum,
  ArchivalStorageResourceApiEnum,
} from "@shared/enums/api.enum";
import {
  AppRoutesEnum,
  PreservationPlanningRoutesEnum,
  PreservationSpaceRoutesEnum,
  SharedAipRoutesEnum,
} from "@shared/enums/routes.enum";
import {
  isNullOrUndefined,
  isNumberReal,
  StringUtil,
  urlSeparator,
} from "solidify-frontend";

export class AipHelper {
  static generateUrlResourceHistoryDataFile(store: Store, storagionNumber: number | undefined = undefined, url: string | undefined = undefined): string {
    if (isNullOrUndefined(store)) {
      return StringUtil.stringEmpty;
    }

    url = isNullOrUndefined(url) ? store.selectSnapshot(state => state.router?.state?.url) : url;

    const urlParts = url.split(AppRoutesEnum.separator);
    const aipId = urlParts[urlParts.indexOf(SharedAipRoutesEnum.aipDetail) + 1];
    const subResourceUrl = urlSeparator + aipId + urlSeparator + ApiResourceNameEnum.DATAFILE;
    return this.generateUrlResource(store, storagionNumber, url) + subResourceUrl;
  }

  static generateUrlResource(store: Store, storagionNumber: number | undefined = undefined, url: string | undefined = undefined): string {
    if (isNullOrUndefined(store)) {
      return StringUtil.stringEmpty;
    }

    url = isNullOrUndefined(url) ? store.selectSnapshot(state => state.router?.state?.url) : url;

    if (url.includes(PreservationPlanningRoutesEnum.aipDownloaded)) {
      return AccessResourceApiEnum.aipDownloaded;
    }

    if (url.includes(PreservationSpaceRoutesEnum.aipSteward)) {
      return ArchivalStorageResourceApiEnum.aip;
    }

    if (isNullOrUndefined(storagionNumber)) {
      if (url.includes(PreservationPlanningRoutesEnum.job)) {
      } else if (url.includes(SharedAipRoutesEnum.aipDetail)) {
        const urlParts = url.split(AppRoutesEnum.separator);
        const partBeforeAipDetail = urlParts[urlParts.indexOf(SharedAipRoutesEnum.aipDetail) - 1];
        storagionNumber = +partBeforeAipDetail;
      } else {
        storagionNumber = +store.selectSnapshot(state => state.router.state.root.children[0].children[0].children[0].params[PreservationPlanningRoutesEnum.storagionNumberWithoutPrefixParam]);
      }
    }

    if (!isNumberReal(storagionNumber)) {
      storagionNumber = environment.defaultStorageIndex + 1;
    }

    const storagion = ArchivalStorageResourceApiEnum.aipStorages.find(aip => aip.index === +storagionNumber);
    if (isNullOrUndefined(storagion)) {
      throw new Error(`The storagion index '${storagionNumber}' is not find in setting`);
    }
    return storagion.url;
  }

  static determineMode(store: Store): AipMode {
    const url = store.selectSnapshot(state => state.router.state.url);
    if (url.includes(PreservationPlanningRoutesEnum.aipDownloaded)) {
      return AipMode.DOWNLOADED_AIP;
    }
    if (url.includes(PreservationSpaceRoutesEnum.aipSteward)) {
      return AipMode.AIP_STEWARD;
    }
    return AipMode.AIP;
  }

  static getStoragionNameByIndex(index: number): string {
    if (environment.archivalStorageName.length < index) {
      return String(index);
    }
    return environment.archivalStorageName[index - 1];
  }
}

export enum AipMode {
  AIP = "AIP",
  DOWNLOADED_AIP = "DOWNLOADED_AIP",
  AIP_STEWARD = "AIP_STEWARD",
}
