import {
  ChangeDetectionStrategy,
  ChangeDetectorRef,
  Component,
  OnDestroy,
  OnInit,
} from "@angular/core";
import {MatDialog} from "@angular/material/dialog";
import {
  ActivatedRoute,
  NavigationCancel,
  NavigationEnd,
  Router,
} from "@angular/router";
import {appAuthorizedOrganizationalUnitNameSpace} from "@app/stores/authorized-organizational-unit/app-authorized-organizational-unit.action";
import {AppAuthorizedOrganizationalUnitState} from "@app/stores/authorized-organizational-unit/app-authorized-organizational-unit.state";
import {Enums} from "@enums";
import {environment} from "@environments/environment";
import {
  Aip,
  OrganizationalUnit,
} from "@models";
import {Navigate} from "@ngxs/router-plugin";
import {
  Actions,
  Store,
} from "@ngxs/store";
import {SharedAbstractListRoutable} from "@shared/components/routables/shared-abstract-list/shared-abstract-list.routable";
import {DataTableComponentEnum} from "@shared/enums/data-table-component.enum";
import {FieldTypeEnum} from "@shared/enums/field-type.enum";
import {LabelTranslateEnum} from "@shared/enums/label-translate.enum";
import {LocalStateEnum} from "@shared/enums/local-state.enum";
import {
  PreservationPlanningRoutesEnum,
  RoutesEnum,
  SharedAipRoutesEnum,
  urlSeparator,
} from "@shared/enums/routes.enum";
import {
  AipHelper,
  AipMode,
} from "@shared/features/aip/helpers/aip.helper";
import {
  SharedAipAction,
  sharedAipActionNameSpace,
} from "@shared/features/aip/stores/shared-aip.action";
import {
  SharedAipState,
  SharedAipStateModel,
} from "@shared/features/aip/stores/shared-aip.state";
import {RouterExtService} from "@shared/services/router-ext.service";
import {Observable} from "rxjs";
import {
  distinctUntilChanged,
  filter,
  tap,
} from "rxjs/operators";
import {
  isNotNullNorUndefined,
  isNullOrUndefined,
  MappingObjectUtil,
  MemoizedUtil,
  OrderEnum,
  Override,
  OverrideProperty,
  QueryParametersUtil,
  ResourceNameSpace,
  Sort,
} from "solidify-frontend";

@Component({
  selector: "dlcm-shared-aip-list-routable",
  templateUrl: "../../../../../components/routables/shared-abstract-list/shared-abstract-list.routable.html",
  styleUrls: ["./shared-aip-list.routable.scss"],
  changeDetection: ChangeDetectionStrategy.OnPush,
})
export class SharedAipListRoutable extends SharedAbstractListRoutable<Aip, SharedAipStateModel> implements OnInit, OnDestroy {
  KEY_BACK_BUTTON: string | undefined = LabelTranslateEnum.back;
  readonly KEY_CREATE_BUTTON: string = undefined;
  readonly KEY_PARAM_NAME: keyof Aip & string = undefined;

  appAuthorizedOrganizationalUnitSort: Sort<OrganizationalUnit> = {
    field: "name",
    order: OrderEnum.ascending,
  };
  appAuthorizedOrganizationalUnitNameSpace: ResourceNameSpace = appAuthorizedOrganizationalUnitNameSpace;
  appAuthorizedOrganizationalUnitState: typeof AppAuthorizedOrganizationalUnitState = AppAuthorizedOrganizationalUnitState;

  mode: AipMode;

  stickyTopPosition: number = environment.defaultStickyDatatableHeight - 1;

  @OverrideProperty()
  skipInitialQuery: boolean = true;

  private readonly KEY_ARCHIVAL_UNIT: keyof Aip = "archivalUnit";

  columnsSkippedToClear: string[] = [
    this.KEY_ARCHIVAL_UNIT,
  ];

  storagion_number: number | undefined = undefined;

  archivalUnit(value: SharedAipRoutesEnum): void {
    if (value === SharedAipRoutesEnum.aipTabUnit) {
      this.getAipListWithParameter(true);
    } else if (value === SharedAipRoutesEnum.aipTabCollections) {
      this.getAipListWithParameter(false);
    } else {
      this.getAipListWithParameter(undefined);
    }
  }

  constructor(protected readonly store: Store,
              protected readonly _route: ActivatedRoute,
              protected readonly _routerExt: RouterExtService,
              protected readonly _changeDetector: ChangeDetectorRef,
              protected readonly _actions$: Actions,
              protected readonly _dialog: MatDialog,
              protected readonly _router: Router) {
    super(store, _changeDetector, _route, _routerExt, _actions$, _dialog, LocalStateEnum.shared_aip, sharedAipActionNameSpace, {
      canCreate: false,
    }, LocalStateEnum.shared);
    this.cleanIsNeeded = true;
  }

  ngOnInit(): void {
    //only clean the query-parameters if coming from the main menu but not from aip-detail since it already calls the clean in ngOnDestroy()
    if (isNotNullNorUndefined(this._routerExt.getPreviousUrl())) {
      if (this._routerExt.getPreviousUrl().includes(SharedAipRoutesEnum.aipDetail)) {
        this.cleanIsNeeded = false;
      } else {
        this.clean();
      }
    }
    super.ngOnInit();
    this.mode = AipHelper.determineMode(this._store);
    this._computeBackButtonLabel();
    this.updateListWithCurrentTab();
    this.subscribe(this.observeRoutingUpdate());
    if (this.mode === AipMode.AIP) {
      this.storagion_number = +this._route.snapshot.parent.parent.paramMap.get(PreservationPlanningRoutesEnum.storagionNumberWithoutPrefixParam);
    }
  }

  private _computeBackButtonLabel(): void {
    switch (this.mode) {
      case AipMode.AIP:
        this.KEY_BACK_BUTTON = LabelTranslateEnum.backToStoragionList;
        break;
      case AipMode.DOWNLOADED_AIP:
        this.KEY_BACK_BUTTON = LabelTranslateEnum.backToPreservationPlanning;
        break;
      case AipMode.AIP_STEWARD:
      default:
        this.KEY_BACK_BUTTON = LabelTranslateEnum.back;
        break;
    }
  }

  private observeRoutingUpdate(): Observable<any> {
    return this._router.events
      .pipe(
        filter(event => event instanceof NavigationCancel || event instanceof NavigationEnd),
        distinctUntilChanged(),
        tap(event => {
          this.updateListWithCurrentTab();
        }),
      );
  }

  private updateListWithCurrentTab(): void {
    const tab = this._route.snapshot.paramMap.get(SharedAipRoutesEnum.tabWithoutPrefixParam);
    this.archivalUnit(tab as SharedAipRoutesEnum);
  }

  @Override()
  showDetail(model: Aip): void {
    if (this.mode === AipMode.AIP) {
      this.store.dispatch(new Navigate([RoutesEnum.preservationPlanningAip + urlSeparator + this.storagion_number + urlSeparator + SharedAipRoutesEnum.aipDetail, model.resId]));
    } else {
      this.store.dispatch(new Navigate([RoutesEnum.preservationPlanningAipDownloadedDetail, model.resId]));
    }
  }

  defineColumns(): void {
    this.columns = [
      {
        field: "info.name" as any,
        header: LabelTranslateEnum.nameLabel,
        type: FieldTypeEnum.string,
        order: OrderEnum.none,
        isFilterable: true,
        isSortable: true,
      },
      {
        field: "smartSize",
        header: LabelTranslateEnum.size,
        type: FieldTypeEnum.string,
        order: OrderEnum.none,
        filterableField: "archiveSize",
        sortableField: "archiveSize",
        isFilterable: true,
        isSortable: true,
        width: "100px",
      },
      {
        field: "archiveFileNumber",
        header: LabelTranslateEnum.files,
        type: FieldTypeEnum.string,
        order: OrderEnum.none,
        filterableField: "archiveFileNumber",
        isFilterable: true,
        isSortable: true,
        width: "80px",
      },
      {
        field: "info.status" as any,
        header: LabelTranslateEnum.status,
        type: FieldTypeEnum.singleSelect,
        order: OrderEnum.none,
        isFilterable: true,
        isSortable: true,
        translate: true,
        filterEnum: Enums.Package.StatusEnumTranslate,
        component: DataTableComponentEnum.status,
      },
      {
        field: "info.organizationalUnitId" as any,
        header: LabelTranslateEnum.organizationalUnit,
        type: FieldTypeEnum.searchableSingleSelect,
        order: OrderEnum.none,
        component: DataTableComponentEnum.organizationalUnitName,
        isFilterable: true,
        isSortable: true,
        resourceNameSpace: this.appAuthorizedOrganizationalUnitNameSpace,
        resourceState: this.appAuthorizedOrganizationalUnitState as any,
        searchableSingleSelectSort: this.appAuthorizedOrganizationalUnitSort,
      },
      {
        field: "creation.when" as any,
        header: LabelTranslateEnum.created,
        type: FieldTypeEnum.datetime,
        order: OrderEnum.none,
        isFilterable: false,
        isSortable: true,
      },
      {
        field: "lastUpdate.when" as any,
        header: LabelTranslateEnum.updated,
        type: FieldTypeEnum.datetime,
        order: OrderEnum.descending,
        isFilterable: false,
        isSortable: true,
      },
    ];
  }

  getAipListWithParameter(isArchivalUnit: boolean | undefined): void {
    let queryParameter = MemoizedUtil.queryParametersSnapshot(this.store, SharedAipState);
    queryParameter = QueryParametersUtil.clone(queryParameter);
    if (isNullOrUndefined(queryParameter.sort?.field)) {
      const columnToSort = this.columns.find(c => c.order !== OrderEnum.none);
      if (isNotNullNorUndefined(columnToSort)) {
        queryParameter.sort = {
          field: columnToSort.field,
          order: columnToSort.order,
        };
      }
    }
    if (isNullOrUndefined(isArchivalUnit)) {
      MappingObjectUtil.delete(queryParameter.search.searchItems, this.KEY_ARCHIVAL_UNIT);
    } else {
      MappingObjectUtil.set(queryParameter.search.searchItems, this.KEY_ARCHIVAL_UNIT, isArchivalUnit.toString());
    }
    this.store.dispatch(new SharedAipAction.GetAll(queryParameter, false));
  }

  conditionDisplayEditButton(model: Aip): boolean {
    return false;
  }

  conditionDisplayDeleteButton(model: Aip): boolean {
    return false;
  }

  back(): void {
    if (this.mode === AipMode.AIP) {
      this._store.dispatch(new Navigate([RoutesEnum.preservationPlanningAip]));
    } else {
      this._store.dispatch(new Navigate([RoutesEnum.preservationPlanning]));
    }
  }

  ngOnDestroy(): void {
    super.ngOnDestroy();
  }
}
