import {
  ChangeDetectionStrategy,
  ChangeDetectorRef,
  Component,
  OnInit,
  ViewChild,
} from "@angular/core";
import {MatDialog} from "@angular/material/dialog";
import {ActivatedRoute} from "@angular/router";
import {Enums} from "@enums";
import {Aip} from "@models";
import {Navigate} from "@ngxs/router-plugin";
import {
  Actions,
  Store,
} from "@ngxs/store";
import {SharedAbstractDetailEditRoutable} from "@shared/components/routables/shared-abstract-detail-edit/shared-abstract-detail-edit.routable";
import {DataTableComponentEnum} from "@shared/enums/data-table-component.enum";
import {FieldTypeEnum} from "@shared/enums/field-type.enum";
import {IconNameEnum} from "@shared/enums/icon-name.enum";
import {LabelTranslateEnum} from "@shared/enums/label-translate.enum";
import {LocalStateEnum} from "@shared/enums/local-state.enum";
import {AppRoutesEnum} from "@shared/enums/routes.enum";
import {SharedAipFormPresentational} from "@shared/features/aip/components/presentationals/aip-form/shared-aip-form.presentational";
import {AipDataFile} from "@shared/features/aip/models/aip-data-file.model";
import {SharedAipDataFileAction} from "@shared/features/aip/stores/aip-data-file/shared-aip-data-file.action";
import {SharedAipDataFileState} from "@shared/features/aip/stores/aip-data-file/shared-aip-data-file.state";
import {sharedAipActionNameSpace} from "@shared/features/aip/stores/shared-aip.action";
import {SharedAipStateModel} from "@shared/features/aip/stores/shared-aip.state";
import {DataTableActions} from "@shared/models/data-table-actions.model";
import {DataTableColumns} from "@shared/models/data-table-columns.model";
import {LocalStateModel} from "@shared/models/local-state.model";
import {Observable} from "rxjs";
import {
  isNotNullNorUndefined,
  MemoizedUtil,
  OrderEnum,
  QueryParameters,
} from "solidify-frontend";

@Component({
  selector: "dlcm-shared-aip-file-routable",
  templateUrl: "./shared-aip-file.routable.html",
  styleUrls: ["./shared-aip-file.routable.scss"],
  changeDetection: ChangeDetectionStrategy.OnPush,
})
export class SharedAipFileRoutable extends SharedAbstractDetailEditRoutable<AipDataFile, SharedAipStateModel> implements OnInit {
  isLoadingDataFileObs: Observable<boolean> = MemoizedUtil.isLoading(this._store, SharedAipDataFileState);
  listDataFileObs: Observable<Aip[]> = MemoizedUtil.list(this._store, SharedAipDataFileState);
  queryParametersObs: Observable<QueryParameters> = MemoizedUtil.queryParameters(this._store, SharedAipDataFileState);

  isInDetailMode: boolean = false;

  @ViewChild("formPresentational")
  readonly formPresentational: SharedAipFormPresentational;

  columns: DataTableColumns<AipDataFile>[];
  actions: DataTableActions<AipDataFile>[];

  readonly KEY_PARAM_NAME: keyof AipDataFile & string = undefined;

  constructor(protected readonly _store: Store,
              protected readonly _route: ActivatedRoute,
              protected readonly _actions$: Actions,
              protected readonly _changeDetector: ChangeDetectorRef,
              protected readonly _dialog: MatDialog) {
    super(_store, _route, _actions$, _changeDetector, _dialog, LocalStateEnum.shared_aip, sharedAipActionNameSpace, LocalStateEnum.shared);

    this.columns = [
      {
        field: "fileName",
        header: LabelTranslateEnum.fileName,
        type: FieldTypeEnum.string,
        order: OrderEnum.none,
        isSortable: true,
        isFilterable: false,
      },
      {
        field: "creation.when" as any,
        header: LabelTranslateEnum.created,
        type: FieldTypeEnum.datetime,
        order: OrderEnum.none,
        isFilterable: false,
        isSortable: true,
      },
      {
        field: "status",
        header: LabelTranslateEnum.status,
        type: FieldTypeEnum.singleSelect,
        order: OrderEnum.none,
        isSortable: false,
        isFilterable: false,
        translate: true,
        filterEnum: Enums.DataFile.StatusEnumTranslate,
        component: DataTableComponentEnum.status,
      },
      {
        field: "smartSize",
        header: LabelTranslateEnum.size,
        type: FieldTypeEnum.string,
        order: OrderEnum.none,
        sortableField: "dataFile.fileSize" as any,
        filterableField: "dataFile.fileSize" as any,
        isSortable: false,
        isFilterable: false,
        translate: true,
        width: "100px",
      },
      {
        field: "complianceLevel",
        header: LabelTranslateEnum.complianceLevel,
        type: FieldTypeEnum.singleSelect,
        order: OrderEnum.none,
        isSortable: false,
        isFilterable: false,
        translate: true,
        component: DataTableComponentEnum.conformityLevelStar,
        filterEnum: Enums.DataFile.ComplianceLevelEnumTranslate,
      },
    ];

    this.actions = [
      {
        logo: IconNameEnum.download,
        callback: (aipDataFile: AipDataFile) => this.downloadFile(this._resId, aipDataFile),
        placeholder: current => LabelTranslateEnum.download,
        displayOnCondition: (aip: Aip) => true,
      },
      {
        logo: IconNameEnum.resume,
        callback: (aipDataFile: AipDataFile) => this.resumeFile(this._resId, aipDataFile),
        placeholder: current => LabelTranslateEnum.resume,
        displayOnCondition: (aipDataFile: AipDataFile) => isNotNullNorUndefined(aipDataFile) && aipDataFile.status === Enums.DataFile.StatusEnum.IN_ERROR,
      },
    ];
  }

  ngOnInit(): void {
    super.ngOnInit();
    this.getCurrentModelOnParent();
  }

  private getCurrentModelOnParent(): void {
    this._resId = this._route.snapshot.parent.paramMap.get(AppRoutesEnum.paramIdWithoutPrefixParam);
    this.getAipById(this._resId);
  }

  private getAipById(id: string): void {
    this.getSubResourceWithParentId(id);
  }

  getSubResourceWithParentId(id: string): void {
    this._store.dispatch(new SharedAipDataFileAction.GetAll(id, null, true));
    // this._store.dispatch(new PreservationSipDataFileAction.GetListFolder(id));
  }

  onQueryParametersEvent(queryParameters: QueryParameters): void {
    this._store.dispatch(new SharedAipDataFileAction.ChangeQueryParameters(this._resId, queryParameters, true));
  }

  refresh(): void {
    this._store.dispatch(new SharedAipDataFileAction.Refresh(this._resId));
  }

  download($event: AipDataFile): void {
    this._store.dispatch(new SharedAipDataFileAction.Download(this._resId, $event));
  }

  showDetail(dataFile: AipDataFile): void {
    this._store.dispatch(new Navigate([this._store.selectSnapshot((s: LocalStateModel) => s.router.state.url), dataFile.resId]));
  }

  private downloadFile(parentId: string, dataFile: AipDataFile): void {
    this._store.dispatch(new SharedAipDataFileAction.Download(parentId, dataFile));
  }

  private resumeFile(parentId: string, dataFile: AipDataFile): void {
    this._store.dispatch(new SharedAipDataFileAction.Resume(parentId, dataFile));
  }
}
