import {Location} from "@angular/common";
import {
  ChangeDetectionStrategy,
  ChangeDetectorRef,
  Component,
  OnInit,
  ViewChild,
} from "@angular/core";
import {MatDialog} from "@angular/material/dialog";
import {MatTabGroup} from "@angular/material/tabs";
import {
  ActivatedRoute,
  NavigationCancel,
  NavigationEnd,
  Router,
} from "@angular/router";
import {Enums} from "@enums";
import {environment} from "@environments/environment";
import {Aip} from "@models";
import {Navigate} from "@ngxs/router-plugin";
import {
  Actions,
  Select,
  Store,
} from "@ngxs/store";
import {Tab} from "@shared/components/containers/shared-tabs/shared-tabs.container";
import {SharedHistoryDialog} from "@shared/components/dialogs/shared-history/shared-history.dialog";
import {SharedAbstractDetailEditCommonRoutable} from "@shared/components/routables/shared-abstract-detail-edit-common/shared-abstract-detail-edit-common.routable";
import {IconNameEnum} from "@shared/enums/icon-name.enum";
import {LabelTranslateEnum} from "@shared/enums/label-translate.enum";
import {LocalStateEnum} from "@shared/enums/local-state.enum";
import {
  PreservationPlanningRoutesEnum,
  RoutesEnum,
  SharedAipRoutesEnum,
  urlSeparator,
} from "@shared/enums/routes.enum";
import {
  SharedAipApproveDisposalDialog,
  SharedAipExtendRetentionDialogMode,
} from "@shared/features/aip/components/dialogs/aip-approve-disposal/shared-aip-approve-disposal.dialog";
import {
  SharedAipExtendRetentionDialog,
  SharedAipExtendRetentionDialogData,
} from "@shared/features/aip/components/dialogs/aip-extend-retention/shared-aip-extend-retention.dialog";
import {
  AipHelper,
  AipMode,
} from "@shared/features/aip/helpers/aip.helper";
import {
  SharedAipAction,
  sharedAipActionNameSpace,
} from "@shared/features/aip/stores/shared-aip.action";
import {
  SharedAipState,
  SharedAipStateModel,
} from "@shared/features/aip/stores/shared-aip.state";
import {SharedAipStatusHistoryAction} from "@shared/features/aip/stores/status-history/shared-aip-status-history.action";
import {SharedAipStatusHistoryState} from "@shared/features/aip/stores/status-history/shared-aip-status-history.state";
import {ExtraButtonToolbar} from "@shared/models/extra-button-toolbar.model";
import {StatusHistoryDialog} from "@shared/models/status-history-dialog.model";
import {StatusHistory} from "@shared/models/status-history.model";
import {SecurityService} from "@shared/services/security.service";
import {Observable} from "rxjs";
import {
  distinctUntilChanged,
  filter,
  map,
  take,
  tap,
} from "rxjs/operators";
import {
  isFalse,
  isNullOrUndefined,
  MappingObjectUtil,
  MemoizedUtil,
  OverrideProperty,
  QueryParameters,
  QueryParametersUtil,
} from "solidify-frontend";

@Component({
  selector: "dlcm-shared-aip-detail-edit-routable",
  templateUrl: "./shared-aip-detail-edit.routable.html",
  styleUrls: ["./shared-aip-detail-edit.routable.scss"],
  changeDetection: ChangeDetectionStrategy.OnPush,
})
export class SharedAipDetailEditRoutable extends SharedAbstractDetailEditCommonRoutable<Aip, SharedAipStateModel> implements OnInit {
  @Select(SharedAipState.isLoadingWithDependency) isLoadingWithDependencyObs: Observable<boolean>;
  @Select(SharedAipState.isReadyToBeDisplayed) isReadyToBeDisplayedObs: Observable<boolean>;

  historyObs: Observable<StatusHistory[]> = MemoizedUtil.select(this._store, SharedAipStatusHistoryState, state => state.history);
  isLoadingHistoryObs: Observable<boolean> = MemoizedUtil.isLoading(this._store, SharedAipStatusHistoryState);
  queryParametersObs: Observable<QueryParameters> = MemoizedUtil.select(this._store, SharedAipStatusHistoryState, state => state.queryParameters);

  @OverrideProperty()
  getByIdIfAlreadyInState: boolean = false;

  readonly KEY_PARAM_NAME: keyof Aip & string = "archiveId";

  @OverrideProperty()
  readonly editAvailable: boolean = false;

  @OverrideProperty()
  readonly deleteAvailable: boolean = false;

  @ViewChild("matTabGroup")
  readonly matTabGroup: MatTabGroup;

  storagion_number: number | undefined = undefined;
  mode: AipMode;
  backButtonLabel: string;

  private readonly KEY_ARCHIVAL_UNIT: keyof Aip = "archivalUnit";

  private currentTab: Tab;

  private get rootUrl(): string[] {
    switch (this.mode) {
      case AipMode.AIP:
        return [RoutesEnum.preservationPlanningAip, this.storagion_number + "", SharedAipRoutesEnum.aipDetail, this._resId];
      case AipMode.AIP_STEWARD:
        return [RoutesEnum.preservationSpaceAipStewardDetail, this._resId];
      case AipMode.DOWNLOADED_AIP:
        return [RoutesEnum.preservationPlanningAipDownloadedDetail, this._resId];
    }
    return [];
  }

  listTabs: Tab[] = [
    {
      id: TabEnum.METADATA,
      suffixUrl: SharedAipRoutesEnum.aipMetadata,
      icon: IconNameEnum.metadata,
      titleToTranslate: LabelTranslateEnum.metadata,
      route: () => [...this.rootUrl, SharedAipRoutesEnum.aipMetadata],
    },
    {
      id: TabEnum.FILES,
      suffixUrl: SharedAipRoutesEnum.aipFiles,
      icon: IconNameEnum.files,
      titleToTranslate: LabelTranslateEnum.files,
      route: () => [...this.rootUrl, SharedAipRoutesEnum.aipFiles],
      conditionDisplay: () => this.currentObs.pipe(map(current => current?.dataFileNumber > 0)),
    },
    {
      id: TabEnum.COLLECTION,
      suffixUrl: SharedAipRoutesEnum.aipCollections,
      icon: IconNameEnum.collection,
      titleToTranslate: LabelTranslateEnum.collection,
      route: () => [...this.rootUrl, SharedAipRoutesEnum.aipCollections],
      conditionDisplay: () => this.currentObs.pipe(map(current => current?.collectionSize > 0)),
    },
  ];

  listExtraButtons: ExtraButtonToolbar<Aip>[] = [
    {
      color: "primary",
      icon: IconNameEnum.simpleChecksum,
      displayCondition: current => this.mode === AipMode.AIP && !isNullOrUndefined(current) && !isNullOrUndefined(current.info) && Enums.Package.statusIsCompleted(current.info.status as Enums.Package.StatusEnum),
      callback: () => this.simpleChecksum(),
      labelToTranslate: (current) => LabelTranslateEnum.simpleChecksums,
      order: 40,
    },
    {
      color: "primary",
      icon: IconNameEnum.doubleChecksum,
      displayCondition: current => this.mode === AipMode.AIP && !isNullOrUndefined(current) && !isNullOrUndefined(current.info) && Enums.Package.statusIsCompleted(current.info.status as Enums.Package.StatusEnum),
      callback: () => this.deepChecksum(),
      labelToTranslate: (current) => LabelTranslateEnum.deepChecksums,
      order: 40,
    },
    {
      color: "primary",
      icon: IconNameEnum.reindex,
      displayCondition: current => this.mode === AipMode.AIP && !isNullOrUndefined(current) && !isNullOrUndefined(current.info) && (Enums.Package.statusIsCompleted(current.info.status as Enums.Package.StatusEnum) || current.info.status === Enums.Package.StatusEnum.STORED),
      callback: () => this.reIndex(),
      labelToTranslate: (current) => LabelTranslateEnum.reindex,
      order: 40,
    },
    {
      color: "primary",
      icon: IconNameEnum.check,
      displayCondition: current => this.mode === AipMode.AIP && !isNullOrUndefined(current) && !isNullOrUndefined(current.info) && Enums.Package.statusIsCompleted(current.info.status as Enums.Package.StatusEnum),
      callback: () => this.check(),
      labelToTranslate: (current) => LabelTranslateEnum.check,
      order: 40,
    },
    {
      color: "primary",
      icon: IconNameEnum.dispose,
      displayCondition: current => this.mode === AipMode.AIP && !isNullOrUndefined(current) && !isNullOrUndefined(current.info) && current.info.status === Enums.Package.StatusEnum.COMPLETED,
      callback: () => this.dispose(),
      labelToTranslate: (current) => LabelTranslateEnum.disposeOf,
      order: 40,
    },
    {
      color: "primary",
      icon: IconNameEnum.approveDisposal,
      displayCondition: current => this.mode === AipMode.AIP && !isNullOrUndefined(current) && !isNullOrUndefined(current.info) && current.info.status === Enums.Package.StatusEnum.DISPOSABLE && current.dispositionApproval === false,
      callback: () => this.approveDisposal(),
      labelToTranslate: (current) => LabelTranslateEnum.disposalApproval,
      order: 40,
    },
    {
      color: "primary",
      icon: IconNameEnum.approveDisposalByOrgUnit,
      displayCondition: current => this.mode === AipMode.AIP && !isNullOrUndefined(current) && !isNullOrUndefined(current.info) && current.info.status === Enums.Package.StatusEnum.DISPOSABLE && current.dispositionApproval === true,
      callback: () => this.approveDisposalByOrgunit(),
      labelToTranslate: (current) => LabelTranslateEnum.disposalApproval,
      order: 40,
    },
    {
      color: "primary",
      icon: IconNameEnum.approveDisposalByOrgUnit,
      displayCondition: current => this.mode === AipMode.AIP_STEWARD && !isNullOrUndefined(current) && !isNullOrUndefined(current.info) && current.info.status === Enums.Package.StatusEnum.DISPOSABLE,
      callback: () => this.approveDisposalByOrgunit(),
      labelToTranslate: (current) => LabelTranslateEnum.disposalApproval,
      order: 40,
    },
    {
      color: "primary",
      icon: IconNameEnum.extendRetention,
      displayCondition: current => (this.mode === AipMode.AIP_STEWARD || this.mode === AipMode.AIP) && !isNullOrUndefined(current) && !isNullOrUndefined(current.info) && current.info.status === Enums.Package.StatusEnum.DISPOSABLE,
      callback: () => this.extendRetention(),
      labelToTranslate: (current) => LabelTranslateEnum.extendTheRetention,
      order: 40,
    },
    {
      color: "primary",
      icon: IconNameEnum.download,
      displayCondition: current => this.mode !== AipMode.AIP_STEWARD && !isNullOrUndefined(current) && !isNullOrUndefined(current.info) && Enums.Package.statusIsCompleted(current.info.status as Enums.Package.StatusEnum),
      callback: () => this.download(),
      labelToTranslate: (current) => LabelTranslateEnum.download,
      order: 40,
    },
    {
      color: "primary",
      icon: IconNameEnum.resume,
      displayCondition: current => (this.mode === AipMode.AIP || this.mode === AipMode.DOWNLOADED_AIP) && !isNullOrUndefined(current) && !isNullOrUndefined(current.info) && current.info.status === Enums.Package.StatusEnum.IN_ERROR,
      callback: () => this.resume(),
      labelToTranslate: (current) => LabelTranslateEnum.resume,
      order: 40,
    },
    {
      color: "primary",
      icon: IconNameEnum.delete,
      displayCondition: current => !isNullOrUndefined(current) && !isNullOrUndefined(current.info) && this._securityService.isRoot() && this.mode === AipMode.DOWNLOADED_AIP,
      callback: () => this.delete(),
      labelToTranslate: (current) => LabelTranslateEnum.delete,
      order: 40,
    },
  ];

  constructor(protected _store: Store,
              protected route: ActivatedRoute,
              protected readonly _actions$: Actions,
              protected readonly _changeDetector: ChangeDetectorRef,
              protected readonly _router: Router,
              protected readonly dialog: MatDialog,
              private readonly _location: Location,
              private readonly _securityService: SecurityService) {
    super(_store, route, _actions$, _changeDetector, dialog, LocalStateEnum.shared_aip, sharedAipActionNameSpace, LocalStateEnum.shared);
  }

  ngOnInit(): void {
    this.mode = AipHelper.determineMode(this._store);
    if (this.mode === AipMode.AIP) {
      this.storagion_number = +this._route.snapshot.parent.paramMap.get(PreservationPlanningRoutesEnum.storagionNumberWithoutPrefixParam);
    }
    super.ngOnInit();
    this._computeBackButtonLabel();

    this.subscribe(this._router.events
      .pipe(
        filter(event => event instanceof NavigationCancel || event instanceof NavigationEnd),
        distinctUntilChanged(),
        tap(event => {
          this.retrieveResIdFromUrl();
        }),
      ),
    );
  }

  getSubResourceWithParentId(id: string): void {
  }

  backToList(): void {
    if (this.mode === AipMode.AIP) {
      this._store.dispatch(new Navigate([RoutesEnum.preservationPlanningAip + urlSeparator + this.storagion_number, SharedAipRoutesEnum.aipList, this._getTabRoute()]));
    } else if (this.mode === AipMode.DOWNLOADED_AIP) {
      this._store.dispatch(new Navigate([RoutesEnum.preservationPlanningAipDownloaded, this._getTabRoute()]));
    } else if (this.mode === AipMode.AIP_STEWARD) {
      this._location.back();
    }
  }

  private _getTabRoute(): string {
    const queryParameters = MemoizedUtil.selectSnapshot(this._store, SharedAipState, state => state.queryParameters);
    const searchItems = QueryParametersUtil.getSearchItems(queryParameters);
    const archivalUnit = MappingObjectUtil.get(searchItems, this.KEY_ARCHIVAL_UNIT);
    if (archivalUnit === "true") {
      return SharedAipRoutesEnum.aipTabUnit;
    } else if (archivalUnit === "false") {
      return SharedAipRoutesEnum.aipTabCollections;
    } else {
      return SharedAipRoutesEnum.aipTabAll;
    }
  }

  private _computeBackButtonLabel(): void {
    switch (this.mode) {
      case AipMode.AIP_STEWARD:
        this.backButtonLabel = LabelTranslateEnum.backToNotification;
        break;
      case AipMode.DOWNLOADED_AIP:
      case AipMode.AIP:
      default:
        this.backButtonLabel = LabelTranslateEnum.backToList;
        break;
    }
  }

  simpleChecksum(): void {
    this._store.dispatch(new SharedAipAction.SimpleChecksum(this._resId));
  }

  deepChecksum(): void {
    this._store.dispatch(new SharedAipAction.DeepChecksum(this._resId));
  }

  reIndex(): void {
    this._store.dispatch(new SharedAipAction.Reindex(this._resId));
  }

  check(): void {
    this._store.dispatch(new SharedAipAction.Check(this._resId));
  }

  dispose(): void {
    this._store.dispatch(new SharedAipAction.Dispose(this._resId));
  }

  delete(): void {
    this._store.dispatch(new SharedAipAction.Delete(this._resId));
  }

  approveDisposal(): void {
    this.subscribe(this._dialog.open(SharedAipApproveDisposalDialog, {
      minWidth: "500px",
      data: {
        resId: this._resId,
        mode: SharedAipExtendRetentionDialogMode.disposal,
      } as SharedAipExtendRetentionDialogData,
    }).afterClosed().pipe(
      take(1),
      tap((success: string | undefined) => {
        if (isNullOrUndefined(success) || isFalse(success)) {
          return;
        }
        //refresh here ?
      }),
    ));
  }

  approveDisposalByOrgunit(): void {
    this.subscribe(this._dialog.open(SharedAipApproveDisposalDialog, {
      minWidth: "500px",
      data: {
        resId: this._resId,
        mode: SharedAipExtendRetentionDialogMode.disposalByOrgunit,
      } as SharedAipExtendRetentionDialogData,
    }).afterClosed().pipe(
      take(1),
      tap((success: string | undefined) => {
        if (isNullOrUndefined(success) || isFalse(success)) {
          return;
        }
        //refresh here ?
      }),
    ));
  }

  extendRetention(): void {
    this.subscribe(this._dialog.open(SharedAipExtendRetentionDialog, {
      minWidth: "500px",
      data: {
        resId: this._resId,
      } as SharedAipExtendRetentionDialogData,
    }).afterClosed().pipe(
      take(1),
      tap((message: string | undefined) => {
        if (isNullOrUndefined(message)) {
          return;
        }
        //refresh here ?
      }),
    ));
  }

  resume(): void {
    this._store.dispatch(new SharedAipAction.Resume(this._resId));
  }

  download(): void {
    this._store.dispatch(new SharedAipAction.Download(this._resId));
  }

  setCurrentTab($event: Tab): void {
    this.currentTab = $event;
  }

  showHistory(): void {
    this._dialog.open(SharedHistoryDialog, {
      width: environment.modalWidth,
      data: {
        parentId: null,
        resourceResId: this._resId,
        name: this._state,
        statusHistory: this.historyObs,
        isLoading: this.isLoadingHistoryObs,
        queryParametersObs: this.queryParametersObs,
        state: SharedAipStatusHistoryAction,
      } as StatusHistoryDialog,
    });
  }
}

enum TabEnum {
  METADATA = "METADATA",
  FILES = "FILES",
  COLLECTION = "COLLECTIONS",
}
