import {
  ChangeDetectionStrategy,
  ChangeDetectorRef,
  Component,
  OnInit,
  ViewChild,
} from "@angular/core";
import {MatDialog} from "@angular/material/dialog";
import {ActivatedRoute} from "@angular/router";
import {Enums} from "@enums";
import {Aip} from "@models";
import {Navigate} from "@ngxs/router-plugin";
import {
  Actions,
  Store,
} from "@ngxs/store";
import {SharedAbstractDetailEditRoutable} from "@shared/components/routables/shared-abstract-detail-edit/shared-abstract-detail-edit.routable";
import {DataTableComponentEnum} from "@shared/enums/data-table-component.enum";
import {FieldTypeEnum} from "@shared/enums/field-type.enum";
import {IconNameEnum} from "@shared/enums/icon-name.enum";
import {LabelTranslateEnum} from "@shared/enums/label-translate.enum";
import {LocalStateEnum} from "@shared/enums/local-state.enum";
import {
  AppRoutesEnum,
  PreservationPlanningRoutesEnum,
} from "@shared/enums/routes.enum";
import {SharedAipFormPresentational} from "@shared/features/aip/components/presentationals/aip-form/shared-aip-form.presentational";
import {AipDataFile} from "@shared/features/aip/models/aip-data-file.model";
import {SharedAipCollectionAction} from "@shared/features/aip/stores/aip-collection/shared-aip-collection.action";
import {SharedAipCollectionState} from "@shared/features/aip/stores/aip-collection/shared-aip-collection.state";
import {sharedAipActionNameSpace} from "@shared/features/aip/stores/shared-aip.action";
import {SharedAipStateModel} from "@shared/features/aip/stores/shared-aip.state";
import {DataTableActions} from "@shared/models/data-table-actions.model";
import {DataTableColumns} from "@shared/models/data-table-columns.model";
import {LocalStateModel} from "@shared/models/local-state.model";
import {Observable} from "rxjs";
import {
  MemoizedUtil,
  OrderEnum,
  QueryParameters,
} from "solidify-frontend";

@Component({
  selector: "dlcm-shared-aip-collection-routable",
  templateUrl: "./shared-aip-collection.routable.html",
  styleUrls: ["./shared-aip-collection.routable.scss"],
  changeDetection: ChangeDetectionStrategy.OnPush,
})
export class SharedAipCollectionRoutable extends SharedAbstractDetailEditRoutable<Aip, SharedAipStateModel> implements OnInit {
  isLoadingDataFileObs: Observable<boolean> = MemoizedUtil.isLoading(this._store, SharedAipCollectionState);
  listDataFileObs: Observable<Aip[]> = MemoizedUtil.list(this._store, SharedAipCollectionState);
  queryParametersObs: Observable<QueryParameters> = MemoizedUtil.queryParameters(this._store, SharedAipCollectionState);

  isInDetailMode: boolean = false;

  @ViewChild("formPresentational")
  readonly formPresentational: SharedAipFormPresentational;

  columns: DataTableColumns<Aip>[];
  actions: DataTableActions<Aip>[];

  readonly KEY_PARAM_NAME: keyof Aip & string = undefined;

  constructor(protected readonly _store: Store,
              protected readonly _route: ActivatedRoute,
              protected readonly _actions$: Actions,
              protected readonly _changeDetector: ChangeDetectorRef,
              protected readonly _dialog: MatDialog) {
    super(_store, _route, _actions$, _changeDetector, _dialog, LocalStateEnum.shared_aip, sharedAipActionNameSpace, LocalStateEnum.shared);

    this.columns = [
      {
        field: "info.name" as any,
        header: LabelTranslateEnum.nameLabel,
        type: FieldTypeEnum.string,
        order: OrderEnum.none,
        isSortable: true,
        isFilterable: false,
      },
      {
        field: "creation.when" as any,
        header: LabelTranslateEnum.created,
        type: FieldTypeEnum.datetime,
        order: OrderEnum.none,
        isFilterable: false,
        isSortable: true,
      },
      {
        field: "info.status" as any,
        header: LabelTranslateEnum.status,
        type: FieldTypeEnum.singleSelect,
        order: OrderEnum.none,
        isSortable: false,
        isFilterable: false,
        translate: true,
        filterEnum: Enums.DataFile.StatusEnumTranslate,
        component: DataTableComponentEnum.status,
      },
      {
        field: "smartSize",
        header: LabelTranslateEnum.size,
        type: FieldTypeEnum.string,
        order: OrderEnum.none,
        sortableField: "dataFile.fileSize" as any,
        filterableField: "dataFile.fileSize" as any,
        isSortable: false,
        isFilterable: false,
        translate: true,
        width: "100px",
      },
      {
        field: "info.complianceLevel" as any,
        header: LabelTranslateEnum.complianceLevel,
        type: FieldTypeEnum.singleSelect,
        order: OrderEnum.none,
        isSortable: false,
        isFilterable: false,
        translate: true,
        component: DataTableComponentEnum.conformityLevelStar,
        filterEnum: Enums.DataFile.ComplianceLevelEnumTranslate,
      },
    ];

    this.actions = [
      {
        logo: IconNameEnum.download,
        callback: (aip: Aip) => this.downloadAipAip(this._resId, aip),
        placeholder: current => LabelTranslateEnum.download,
        displayOnCondition: (aip: Aip) => true,
      },
      {
        logo: IconNameEnum.internalLink,
        callback: (aip: Aip) => this._goToAip(aip),
        placeholder: current => LabelTranslateEnum.goToAip,
        displayOnCondition: (aip: Aip) => true,
      },
    ];
  }

  ngOnInit(): void {
    super.ngOnInit();
    this.getCurrentModelOnParent();
  }

  private getCurrentModelOnParent(): void {
    this._resId = this._route.snapshot.parent.paramMap.get(AppRoutesEnum.paramIdWithoutPrefixParam);
    this.getAipById(this._resId);
  }

  private getAipById(id: string): void {
    this.getSubResourceWithParentId(id);
  }

  getSubResourceWithParentId(id: string): void {
    this._store.dispatch(new SharedAipCollectionAction.GetAll(id, null, true));
    // this._store.dispatch(new PreservationSipDataFileAction.GetListFolder(id));
  }

  onQueryParametersEvent(queryParameters: QueryParameters): void {
    this._store.dispatch(new SharedAipCollectionAction.ChangeQueryParameters(this._resId, queryParameters, true));
  }

  refresh(): void {
    this._store.dispatch(new SharedAipCollectionAction.Refresh(this._resId));
  }

  download($event: AipDataFile): void {
    this._store.dispatch(new SharedAipCollectionAction.DownloadAip(this._resId, $event));
  }

  showDetail(aip: Aip): void {
    this._store.dispatch(new Navigate([this._store.selectSnapshot((s: LocalStateModel) => s.router.state.url), aip.resId]));
  }

  private downloadAipAip(parentId: string, aip: Aip): void {
    this._store.dispatch(new SharedAipCollectionAction.DownloadAip(parentId, aip));
  }

  private _goToAip(aip: Aip): void {
    const storagion_number = this._route.snapshot.parent.parent.paramMap.get(PreservationPlanningRoutesEnum.storagionNumberWithoutPrefixParam);
    this._store.dispatch(new SharedAipCollectionAction.GoToAip(aip, storagion_number));
  }
}
