import {
  ChangeDetectionStrategy,
  ChangeDetectorRef,
  Component,
  ElementRef,
  Input,
} from "@angular/core";
import {FormBuilder} from "@angular/forms";
import {appAuthorizedOrganizationalUnitNameSpace} from "@app/stores/authorized-organizational-unit/app-authorized-organizational-unit.action";
import {AppAuthorizedOrganizationalUnitState} from "@app/stores/authorized-organizational-unit/app-authorized-organizational-unit.state";
import {Enums} from "@enums";
import {
  Aip,
  AipChecksumCheck,
} from "@models";
import {TranslateService} from "@ngx-translate/core";
import {SharedAbstractFormPresentational} from "@shared/components/presentationals/shared-abstract-form/shared-abstract-form.presentational";
import {RoutesEnum} from "@shared/enums/routes.enum";
import {AipMode} from "@shared/features/aip/helpers/aip.helper";
import {BaseFormDefinition} from "@shared/models/base-form-definition.model";
import {KeyValueInfo} from "@shared/models/key-value-info.model";
import {sharedLicenseActionNameSpace} from "@shared/stores/license/shared-license.action";
import {SharedLicenseState} from "@shared/stores/license/shared-license.state";
import {FileUtil} from "@shared/utils/file.util";
import {
  DateUtil,
  EnumUtil,
  isNullOrUndefined,
  KeyValue,
  MARK_AS_TRANSLATABLE,
  PropertyName,
  ResourceNameSpace,
  SolidifyValidator,
  StringUtil,
} from "solidify-frontend";

@Component({
  selector: "dlcm-shared-aip-form",
  templateUrl: "./shared-aip-form.presentational.html",
  styleUrls: ["./shared-aip-form.presentational.scss"],
  changeDetection: ChangeDetectionStrategy.OnPush,
})
export class SharedAipFormPresentational extends SharedAbstractFormPresentational<Aip> {
  accessEnumValues: KeyValue[] = Enums.Deposit.AccessEnumTranslate;
  dataSensitivityEnumValues: KeyValue[] = Enums.Deposit.DataSensitivityEnumTranslate;
  packageStatusEnumValuesTranslate: KeyValue[] = Enums.Package.StatusEnumTranslate;

  get packageStatusEnum(): typeof Enums.Package.StatusEnum { return Enums.Package.StatusEnum; }

  sharedLicenseActionNameSpace: ResourceNameSpace = sharedLicenseActionNameSpace;
  sharedLicenseState: typeof SharedLicenseState = SharedLicenseState;

  appAuthorizedOrganizationalUnitNameSpace: ResourceNameSpace = appAuthorizedOrganizationalUnitNameSpace;
  appAuthorizedOrganizationalUnitState: typeof AppAuthorizedOrganizationalUnitState = AppAuthorizedOrganizationalUnitState;

  public formDefinition: FormComponentFormDefinition = new FormComponentFormDefinition();

  @Input()
  mode: AipMode;

  get aipMode(): typeof AipMode {
    return AipMode;
  }

  getSensitivityTooltip(dataSensitivity: Enums.Deposit.DataSensitivityEnum): string {
    return (EnumUtil.getKeyValue(Enums.Deposit.DataSensitivityEnumTranslate, dataSensitivity) as KeyValueInfo)?.infoToTranslate;
  }

  constructor(protected readonly _changeDetectorRef: ChangeDetectorRef,
              protected readonly _elementRef: ElementRef,
              private readonly _fb: FormBuilder,
              private readonly _translate: TranslateService) {
    super(_changeDetectorRef, _elementRef);
  }

  protected initNewForm(): void {
    this.form = this._fb.group({});
  }

  protected bindFormTo(aip: Aip): void {
    this.form = this._fb.group({
      [this.formDefinition.id]: [aip.resId, []],
      [this.formDefinition.organizationalUnitId]: [aip.info.organizationalUnitId, []],
      [this.formDefinition.name]: [aip.info.name, []],
      [this.formDefinition.description]: [aip.info.description, []],
      [this.formDefinition.yearOfCreation]: [DateUtil.extractYearFromDate(aip.creation.when), []],
      [this.formDefinition.status]: [aip.info.status, []],
      [this.formDefinition.lastchecksumCheck]: [this.extractChecksumCheck(aip.checksumCheck), []],
      [this.formDefinition.access]: [aip.info.access, []],
      [this.formDefinition.dataSensitivity]: [aip.info.dataSensitivity, [SolidifyValidator]],
      [this.formDefinition.licenseId]: [aip.info.licenseId, []],
      [this.formDefinition.complianceLevel]: [aip.info.complianceLevel, []],
      [this.formDefinition.container]: [aip.archiveContainer, []],
      [this.formDefinition.archive]: [aip.archiveId, []],
      [this.formDefinition.size]: [FileUtil.transformFileSize(aip.archiveSize), []],
      [this.formDefinition.files]: [aip.archiveFileNumber, []],
      [this.formDefinition.metadataVersion]: [aip.info.metadataVersion, []],
      [this.formDefinition.retention]: [aip.smartRetention, []],
      [this.formDefinition.retentionEnd]: [DateUtil.convertDateToDateTimeString(new Date(aip.retentionEnd)), []],
      [this.formDefinition.tombstoneSize]: [aip.smartTombstoneSize, []],
      [this.formDefinition.embargoAccess]: [aip.info.embargo?.access, []],
      [this.formDefinition.embargoNumberMonths]: [aip.info.embargo?.months, []],
      [this.formDefinition.embargoStartDate]: [isNullOrUndefined(aip.info.embargo?.startDate) ? "" : DateUtil.convertDateToDateTimeString(new Date(aip.info.embargo?.startDate))],
    });
  }

  protected treatmentBeforeSubmit(aip: Aip): Aip {
    return aip;
  }

  extractChecksumCheck(checksum: AipChecksumCheck): string {
    let lastChecksumCheckNever = StringUtil.stringEmpty;
    let lastChecksumCheckSuccess = StringUtil.stringEmpty;
    let lastChecksumCheckFail = StringUtil.stringEmpty;

    const keyLabelNever = MARK_AS_TRANSLATABLE("aip.form.lastChecksumCheck.never");
    const keyLabelSuccess = MARK_AS_TRANSLATABLE("aip.form.lastChecksumCheck.success");
    const keyLabelFail = MARK_AS_TRANSLATABLE("aip.form.lastChecksumCheck.fail");

    this._translate.get([
      keyLabelNever,
      keyLabelSuccess,
      keyLabelFail,
    ]).subscribe(translation => {
      lastChecksumCheckNever = translation[keyLabelNever];
      lastChecksumCheckSuccess = translation[keyLabelSuccess];
      lastChecksumCheckFail = translation[keyLabelFail];
    });

    if (isNullOrUndefined(checksum)) {
      return lastChecksumCheckNever;
    }

    const strDate = DateUtil.convertToLocalDateDateSimple(checksum.checkDate);
    const str = checksum.checkingSucceed ? lastChecksumCheckSuccess : lastChecksumCheckFail;
    return strDate + " (" + str + ")";
  }

  goToSip(): void {
    this._navigateBS.next([RoutesEnum.preservationPlanningSipDetail, this.model.sipIds[0]]);
  }
}

class FormComponentFormDefinition extends BaseFormDefinition {
  @PropertyName() id: string;
  @PropertyName() organizationalUnitId: string;
  @PropertyName() name: string;
  @PropertyName() description: string;
  @PropertyName() yearOfCreation: string;
  @PropertyName() status: string;
  @PropertyName() lastchecksumCheck: string;
  @PropertyName() access: string;
  @PropertyName() dataSensitivity: string;
  @PropertyName() licenseId: string;
  @PropertyName() complianceLevel: string;
  @PropertyName() container: string;
  @PropertyName() archive: string;
  @PropertyName() size: string;
  @PropertyName() files: string;
  @PropertyName() metadataVersion: string;
  @PropertyName() retention: string;
  @PropertyName() retentionEnd: string;
  @PropertyName() tombstoneSize: string;
  @PropertyName() embargoAccess: string;
  @PropertyName() embargoNumberMonths: string;
  @PropertyName() embargoStartDate: string;
}
