import {
  ChangeDetectionStrategy,
  Component,
  Inject,
  OnInit,
} from "@angular/core";
import {
  AbstractControl,
  FormBuilder,
  FormGroup,
  Validators,
} from "@angular/forms";
import {
  MAT_DIALOG_DATA,
  MatDialogRef,
} from "@angular/material/dialog";
import {
  Actions,
  ofActionCompleted,
  Store,
} from "@ngxs/store";
import {SharedAbstractContainer} from "@shared/components/containers/shared-abstract/shared-abstract.container";
import {SharedAipAction} from "@shared/features/aip/stores/shared-aip.action";
import {SharedAipState} from "@shared/features/aip/stores/shared-aip.state";
import {BaseFormDefinition} from "@shared/models/base-form-definition.model";
import {Observable} from "rxjs";
import {
  take,
  tap,
} from "rxjs/operators";
import {
  FormValidationHelper,
  isTrue,
  MemoizedUtil,
  PropertyName,
  SolidifyValidator,
} from "solidify-frontend";

@Component({
  selector: "dlcm-shared-aip-extend-retention-dialog",
  templateUrl: "./shared-aip-extend-retention.dialog.html",
  styleUrls: ["./shared-aip-extend-retention.dialog.scss"],
  changeDetection: ChangeDetectionStrategy.OnPush,
})
export class SharedAipExtendRetentionDialog extends SharedAbstractContainer implements OnInit {
  isLoadingObs: Observable<boolean> = MemoizedUtil.isLoading(this._store, SharedAipState);

  form: FormGroup;
  formDefinition: FormComponentFormDefinition = new FormComponentFormDefinition();

  constructor(private readonly _store: Store,
              private readonly _actions$: Actions,
              private readonly _dialogRef: MatDialogRef<SharedAipExtendRetentionDialog>,
              @Inject(MAT_DIALOG_DATA) public data: SharedAipExtendRetentionDialogData,
              private readonly _fb: FormBuilder) {
    super();
  }

  ngOnInit(): void {
    super.ngOnInit();
    this.form = this._fb.group({
      [this.formDefinition.duration]: ["", [Validators.required, SolidifyValidator]],
    });
  }

  close(): void {
    this._dialogRef.close();
  }

  onSubmit(): void {
    this._store.dispatch(new SharedAipAction.ExtendRetention(this.data.resId, this.form.get(this.formDefinition.duration).value));

    this.subscribe(this._actions$.pipe(
      ofActionCompleted(SharedAipAction.ExtendRetention),
      take(1),
      tap((result) => {
        if (isTrue(result.result.successful)) {
          this._dialogRef.close();
        }
      }),
    ));
  }

  getFormControl(key: string): AbstractControl {
    return FormValidationHelper.getFormControl(this.form, key);
  }

  get formValidationHelper(): typeof FormValidationHelper {
    return FormValidationHelper;
  }
}

class FormComponentFormDefinition extends BaseFormDefinition {
  @PropertyName() duration: string;
}

export interface SharedAipExtendRetentionDialogData {
  resId: string;
}
