import {
  ChangeDetectionStrategy,
  Component,
  Inject,
  OnInit,
} from "@angular/core";
import {
  AbstractControl,
  FormBuilder,
  FormGroup,
  Validators,
} from "@angular/forms";
import {
  MAT_DIALOG_DATA,
  MatDialogRef,
} from "@angular/material/dialog";
import {
  Actions,
  ofActionCompleted,
  Store,
} from "@ngxs/store";
import {SharedAbstractContainer} from "@shared/components/containers/shared-abstract/shared-abstract.container";
import {SharedAipAction} from "@shared/features/aip/stores/shared-aip.action";
import {SharedAipState} from "@shared/features/aip/stores/shared-aip.state";
import {BaseFormDefinition} from "@shared/models/base-form-definition.model";
import {Observable} from "rxjs";
import {
  take,
  tap,
} from "rxjs/operators";
import {
  FormValidationHelper,
  isEmptyString,
  isNullOrUndefined,
  isTrue,
  MemoizedUtil,
  PropertyName,
  SolidifyValidator,
} from "solidify-frontend";

@Component({
  selector: "dlcm-shared-aip-approve-disposal-dialog",
  templateUrl: "./shared-aip-approve-disposal.dialog.html",
  styleUrls: ["./shared-aip-approve-disposal.dialog.scss"],
  changeDetection: ChangeDetectionStrategy.OnPush,
})
export class SharedAipApproveDisposalDialog extends SharedAbstractContainer implements OnInit {
  isLoadingObs: Observable<boolean> = MemoizedUtil.isLoading(this._store, SharedAipState);

  form: FormGroup;
  formDefinition: FormComponentFormDefinition = new FormComponentFormDefinition();

  constructor(private readonly _store: Store,
              private readonly _actions$: Actions,
              private readonly _dialogRef: MatDialogRef<SharedAipApproveDisposalDialog>,
              @Inject(MAT_DIALOG_DATA) public data: SharedAipExtendRetentionDialogData,
              private readonly _fb: FormBuilder) {
    super();
  }

  ngOnInit(): void {
    super.ngOnInit();
    this.form = this._fb.group({
      [this.formDefinition.reason]: ["", [Validators.required, SolidifyValidator]],
    });
  }

  close(): void {
    this._dialogRef.close();
  }

  onSubmit(): void {
    const reason = this.form.get(this.formDefinition.reason).value;
    if (isNullOrUndefined(reason) || isEmptyString(reason)) {
      return;
    }
    let action = null;
    let actionCompleted = null;
    if (this.data.mode === SharedAipExtendRetentionDialogMode.disposal) {
      action = new SharedAipAction.ApproveDisposal(this.data.resId, reason);
      actionCompleted = SharedAipAction.ApproveDisposalSuccess;
    } else if (this.data.mode === SharedAipExtendRetentionDialogMode.disposalByOrgunit) {
      action = new SharedAipAction.ApproveDisposalByOrgunit(this.data.resId, reason);
      actionCompleted = SharedAipAction.ApproveDisposalByOrgunitSuccess;
    } else {
      throw new Error("Unexpected mode");
    }
    this.subscribe(this._actions$.pipe(
      ofActionCompleted(actionCompleted),
      take(1),
      tap((result) => {
        if (isTrue(result.result.successful)) {
          this._dialogRef.close(true);
        }
      }),
    ));
    this._store.dispatch(action);
  }

  getFormControl(key: string): AbstractControl {
    return FormValidationHelper.getFormControl(this.form, key);
  }

  get formValidationHelper(): typeof FormValidationHelper {
    return FormValidationHelper;
  }
}

class FormComponentFormDefinition extends BaseFormDefinition {
  @PropertyName() reason: string;
}

export interface SharedAipExtendRetentionDialogData {
  resId: string;
  mode: SharedAipExtendRetentionDialogMode;
}

export enum SharedAipExtendRetentionDialogMode {
  disposal,
  disposalByOrgunit
}
