import {
  AfterViewInit,
  ChangeDetectorRef,
  Directive,
  ElementRef,
  OnInit,
  ViewChild,
} from "@angular/core";
import {MatDialog} from "@angular/material/dialog";
import {ActivatedRoute} from "@angular/router";
import {SharedAbstractCrudRoutable} from "@app/shared/components/routables/shared-abstract-crud/shared-abstract-crud.routable";
import {DataTableColumns} from "@app/shared/models/data-table-columns.model";
import {StoreRouteLocalUtil} from "@app/shared/utils/store-route-local.util";
import {environment} from "@environments/environment";
import {Navigate} from "@ngxs/router-plugin";
import {
  Actions,
  ofActionCompleted,
  Store,
} from "@ngxs/store";
import {SharedDataTablePresentational} from "@shared/components/presentationals/shared-data-table/shared-data-table.presentational";
import {IconNameEnum} from "@shared/enums/icon-name.enum";
import {LabelTranslateEnum} from "@shared/enums/label-translate.enum";
import {LocalStateEnum} from "@shared/enums/local-state.enum";
import {DataTableActions} from "@shared/models/data-table-actions.model";
import {ExtraButtonToolbar} from "@shared/models/extra-button-toolbar.model";
import {RouterExtService} from "@shared/services/router-ext.service";
import {RouteUtil} from "@shared/utils/route.util";
import {StoreDialogUtil} from "@shared/utils/store-dialog.util";
import {Observable} from "rxjs";
import {tap} from "rxjs/operators";
import {
  BaseResourceType,
  isNullOrUndefined,
  QueryParameters,
  ResourceActionHelper,
  ResourceNameSpace,
  ResourceStateModel,
  StoreUtil,
} from "solidify-frontend";
import {SharedDeleteDialog} from "../../dialogs/shared-abstract-delete/shared-delete.dialog";

@Directive()
export abstract class SharedAbstractListRoutable<TResourceModel extends BaseResourceType, UResourceStateModel extends ResourceStateModel<TResourceModel>> extends SharedAbstractCrudRoutable<TResourceModel, UResourceStateModel> implements OnInit, AfterViewInit {
  isLoadingObs: Observable<boolean>;
  listObs: Observable<TResourceModel[]>;
  queryParametersObs: Observable<QueryParameters>;

  skipInitialQuery: boolean = false;
  stickyTopPosition: number = environment.defaultStickyDatatableHeight;
  columnsSkippedToClear: string[] = [];

  columns: DataTableColumns<TResourceModel>[];
  actions: DataTableActions<TResourceModel>[];
  listNewId: string[];
  cleanIsNeeded: boolean = true;

  @ViewChild("dataTablePresentational")
  readonly dataTablePresentational: SharedDataTablePresentational<TResourceModel>;

  abstract readonly KEY_CREATE_BUTTON: string = LabelTranslateEnum.create;
  abstract readonly KEY_BACK_BUTTON: string | undefined;
  abstract readonly KEY_PARAM_NAME: keyof TResourceModel & string;

  protected constructor(protected readonly _store: Store,
                        protected readonly _changeDetector: ChangeDetectorRef,
                        protected readonly _route: ActivatedRoute,
                        protected readonly _routerExt: RouterExtService,
                        protected readonly _actions$: Actions,
                        protected readonly _dialog: MatDialog,
                        protected readonly _state: LocalStateEnum,
                        private readonly _resourceNameSpace: ResourceNameSpace,
                        public options?: SharedAbstractListRoutableOption<TResourceModel>,
                        protected readonly _parentState?: LocalStateEnum) {
    super(_store, _state, _parentState);
    this.options = Object.assign(this.getDefaultOptions(), options);
    this.isLoadingObs = this._store.select(s => StoreUtil.isLoadingState(super.getState(s)));
    this.listObs = this._store.select(s => this.getState(s).list);
    this.queryParametersObs = this._store.select(s => this.getState(s).queryParameters);
  }

  ngOnInit(): void {
    super.ngOnInit();
    if (!this._routerExt.previousRouteIsChild() && this.cleanIsNeeded) {
      this.clean();
    }
    this.defineColumns();
    this._applyLastSort();
    this.actions = this.defineActions();
  }

  ngAfterViewInit(): void {
    super.ngAfterViewInit();
    if (isNullOrUndefined(this.options.listExtraButtons)) {
      this.options.listExtraButtons = [];
    }
    if (isNullOrUndefined(this.dataTablePresentational?.buttonCleanAllFilter)) {
      return;
    }
    this.options.listExtraButtons = [...this.options.listExtraButtons, this.dataTablePresentational?.buttonCleanAllFilter];
    this._changeDetector.detectChanges();
  }

  private _applyLastSort(): void {
    const queryParameters = this._store.selectSnapshot(s => this.getState(s).queryParameters);
    const sort = queryParameters.sort;
    if (isNullOrUndefined(sort) || isNullOrUndefined(sort.field)) {
      return;
    }
    const index = this.columns.findIndex(c => c.sortableField === sort.field || c.field === sort.field);
    if (index === -1) {
      return;
    }
    this.columns[index].order = sort.order;
  }

  private getDefaultOptions(): SharedAbstractListRoutableOption<TResourceModel> {
    return {
      canCreate: true,
      canGoBack: true,
      canRefresh: true,
    };
  }

  abstract defineColumns(): void;

  protected defineActions(): DataTableActions<TResourceModel>[] {
    return [
      {
        logo: IconNameEnum.edit,
        callback: (model: TResourceModel) => this.goToEdit(model),
        placeholder: current => LabelTranslateEnum.edit,
        displayOnCondition: (model: TResourceModel) => this.conditionDisplayEditButton(model),
        isWrapped: false,
      },
      {
        logo: IconNameEnum.delete,
        callback: (model: TResourceModel) => this.delete(model),
        placeholder: current => LabelTranslateEnum.delete,
        displayOnCondition: (model: TResourceModel) => this.conditionDisplayDeleteButton(model),
        isWrapped: true,
      },
    ];
  }

  abstract conditionDisplayEditButton(model: TResourceModel | undefined): boolean;

  abstract conditionDisplayDeleteButton(model: TResourceModel | undefined): boolean;

  goToEdit(model: TResourceModel): void {
    this._store.dispatch(new Navigate([StoreRouteLocalUtil.getEditRoute(this._state, model.resId)], {}, {skipLocationChange: true}));
  }

  delete(model: TResourceModel): void {
    const datas = StoreDialogUtil.deleteData(this._state);
    datas.name = model[this.KEY_PARAM_NAME]?.toString();
    datas.resId = model.resId;

    this._dialog.open(SharedDeleteDialog, {
      width: "400px",
      data: datas,
    });
    this.subscribe(this._actions$.pipe(ofActionCompleted(this._resourceNameSpace.DeleteSuccess))
      .pipe(
        tap(result => {
          if (result.result.successful) {
            this.getAll();
          }
        }),
      ));
  }

  create(element: ElementRef): void {
    this.navigateNavigate(new Navigate([StoreRouteLocalUtil.getCreateRoute(this._state)], {}, {skipLocationChange: true}));
  }

  getAll(queryParameters?: QueryParameters): void {
    this._store.dispatch(ResourceActionHelper.getAll(this._resourceNameSpace, queryParameters, true));
  }

  showDetail(model: TResourceModel): void {
    this._store.dispatch(new Navigate([StoreRouteLocalUtil.getDetailRoute(this._state), model.resId]));
  }

  onQueryParametersEvent(queryParameters: QueryParameters): void {
    this._store.dispatch(ResourceActionHelper.changeQueryParameters(this._resourceNameSpace, queryParameters, true));
    this._changeDetector.detectChanges(); // Allow to display spinner the first time
  }

  back(): void {
    this._store.dispatch(new Navigate(RouteUtil.generateFullUrlFromActivatedRouteNormal(this._route.parent.parent)));
  }

  clean(): void {
    this._store.dispatch(ResourceActionHelper.clean(this._resourceNameSpace, false, new QueryParameters()));
  }
}

export interface SharedAbstractListRoutableOption<TResourceModel> {
  canCreate?: boolean | Observable<boolean>;
  canGoBack?: boolean;
  canRefresh?: boolean;
  listExtraButtons?: ExtraButtonToolbar<TResourceModel>[];
}
