import {
  Directive,
  OnInit,
} from "@angular/core";
import {AppState} from "@app/stores/app.state";
import {Enums} from "@enums";
import {TranslateService} from "@ngx-translate/core";
import {Navigate} from "@ngxs/router-plugin";
import {Store} from "@ngxs/store";
import {SharedAbstractPresentational} from "@shared/components/presentationals/shared-abstract/shared-abstract.presentational";
import {ApplicationRolePermissionEnum} from "@shared/enums/application-role-permission.enum";
import {DataTestEnum} from "@shared/enums/data-test.enum";
import {IconNameEnum} from "@shared/enums/icon-name.enum";
import {RoutesEnum} from "@shared/enums/routes.enum";
import {PermissionUtil} from "@shared/utils/permission.util";
import {
  isEmptyString,
  MemoizedUtil,
  StringUtil,
} from "solidify-frontend";

@Directive()
export abstract class SharedAbstractHomeRoutable extends SharedAbstractPresentational implements OnInit {
  searchValueInUrl: string = StringUtil.stringEmpty;

  protected abstract _tiles: SharedHomeTileModel[];
  protected _userRolesObs: Enums.UserApplicationRole.UserApplicationRoleEnum[] = MemoizedUtil.selectSnapshot(this._store, AppState, state => state.userRoles);

  protected constructor(protected readonly _store: Store,
                        protected readonly _translate: TranslateService) {
    super();
  }

  navigate(path: RoutesEnum): void {
    this._store.dispatch(new Navigate([path]));
  }

  ngOnInit(): void {
    this._computeTilesToDisplay();
  }

  visibleTiles: SharedHomeTileModel[];

  private _getVisibleTiles(): SharedHomeTileModel[] {
    return this._tiles.filter(tile => this._displayTile(tile));
  }

  private _displayTile(tile: SharedHomeTileModel): boolean {
    return PermissionUtil.isUserHavePermission(true, tile.permission, this._userRolesObs)
      && (isEmptyString(this.searchValueInUrl) || this._translate.instant(tile.titleToTranslate).toLowerCase().startsWith(this.searchValueInUrl));
  }

  search(searchTerm: string): void {
    this.searchValueInUrl = searchTerm.toLowerCase();
    this._computeTilesToDisplay();
    return;
  }

  private _computeTilesToDisplay(): void {
    this.visibleTiles = this._getVisibleTiles();
  }
}

export interface SharedHomeTileModel {
  avatarIcon: IconNameEnum;
  titleToTranslate: string;
  subtitleToTranslate?: string;
  path: RoutesEnum;
  permission: ApplicationRolePermissionEnum;
  dataTest?: DataTestEnum;
}
