import {
  ChangeDetectorRef,
  Directive,
  OnInit,
  Type,
  ViewChild,
} from "@angular/core";
import {MatDialog} from "@angular/material/dialog";
import {ActivatedRoute} from "@angular/router";
import {Enums} from "@enums";
import {Aip} from "@models";
import {
  Actions,
  ofActionCompleted,
  Store,
} from "@ngxs/store";
import {
  SharedFileAndAipDetailDialogModeEnum,
  SharedFileAndAipInformationContainer,
  SharedFileAndAipInformationDialogData,
} from "@shared/components/containers/shared-file-and-aip-information/shared-file-and-aip-information.container";
import {SharedAbstractRoutable} from "@shared/components/routables/shared-abstract/shared-abstract.routable";
import {IconNameEnum} from "@shared/enums/icon-name.enum";
import {LabelTranslateEnum} from "@shared/enums/label-translate.enum";
import {AppRoutesEnum} from "@shared/enums/routes.enum";
import {DataFile} from "@shared/models/business/data-file.model";
import {ExtraButtonToolbar} from "@shared/models/extra-button-toolbar.model";
import {LocalStateModel} from "@shared/models/local-state.model";
import {StatusHistoryNamespace} from "@shared/stores/status-history/status-history-namespace.model";
import {
  StatusHistoryState,
  StatusHistoryStateModel,
} from "@shared/stores/status-history/status-history.state";
import {isNotNullOrUndefined} from "codelyzer/util/isNotNullOrUndefined";
import {Observable} from "rxjs";
import {
  filter,
  take,
  tap,
} from "rxjs/operators";
import {
  AssociationNameSpace,
  AssociationRemoteNameSpace,
  CompositionNameSpace,
  isNotNullNorUndefined,
  MemoizedUtil,
  MemoizedUtilStateType,
  urlSeparator,
} from "solidify-frontend";

@Directive()
export abstract class SharedAbstractFileAipDetailRoutable<TResource extends DataFile | Aip> extends SharedAbstractRoutable implements OnInit {
  resourceObs: Observable<TResource>;

  mode: SharedFileAndAipDetailDialogModeEnum;
  data: SharedFileAndAipInformationDialogData<TResource>;

  @ViewChild("sharedFileAndAipInformationContainer")
  readonly sharedFileAndAipInformationContainer: SharedFileAndAipInformationContainer<TResource>;

  protected _parentId: string | undefined;
  protected _dataFileId: string | undefined;

  protected _extraActions: ExtraButtonToolbar<TResource>[] = [];

  protected _actionsFile: ExtraButtonToolbar<DataFile>[] = [
    {
      labelToTranslate: (current) => LabelTranslateEnum.download,
      color: "primary",
      icon: IconNameEnum.download,
      callback: current => this._downloadDataFile(),
      displayCondition: current => current.status === Enums.DataFile.StatusEnum.PROCESSED || current.status === Enums.DataFile.StatusEnum.READY || current.status === Enums.DataFile.StatusEnum.VIRUS_CHECKED,
      order: 52,
    },
    {
      labelToTranslate: (current) => LabelTranslateEnum.resume,
      color: "primary",
      icon: IconNameEnum.resume,
      callback: current => this._resume(),
      displayCondition: current => current.status === Enums.DataFile.StatusEnum.IN_ERROR,
      order: 54,
    },
  ];

  protected _actionsAip: ExtraButtonToolbar<Aip>[] = [
    {
      labelToTranslate: (current) => LabelTranslateEnum.goToAip,
      icon: IconNameEnum.internalLink,
      color: "primary",
      callback: current => this._goToAip(),
      order: 52,
    },
  ];

  constructor(protected readonly _store: Store,
              protected readonly _route: ActivatedRoute,
              protected readonly _actions$: Actions,
              protected readonly _dialog: MatDialog,
              protected readonly _changeDetector: ChangeDetectorRef,
              protected readonly _state: MemoizedUtilStateType<any, TResource, any>,
              protected readonly _stateAction: CompositionNameSpace | AssociationNameSpace | AssociationRemoteNameSpace,
              protected readonly _statusHistoryState: Type<StatusHistoryState<StatusHistoryStateModel<TResource>, TResource>>,
              protected readonly _statusHistoryNamespace: StatusHistoryNamespace) {
    super();
    this.resourceObs = MemoizedUtil.current(this._store, _state);
  }

  ngOnInit(): void {
    super.ngOnInit();
    this._retrieveResIdFromUrl();
    this._computeMode();
    if (isNotNullNorUndefined(this.mode)) {
      this.subscribe(this._observableCreateData());
      this._getOrRefreshResource();
    }
  }

  protected _observableCreateData(): Observable<TResource> {
    return this.resourceObs.pipe(
      filter(resource => isNotNullOrUndefined(resource) && resource.resId === this._dataFileId),
      tap(resource => {
        this.data = {
          mode: this.mode,
          dataFile: this.mode === SharedFileAndAipDetailDialogModeEnum.file ? resource : undefined,
          aip: this.mode === SharedFileAndAipDetailDialogModeEnum.aip ? resource : undefined,
          resId: resource.resId,
          parentId: this._parentId,
          statusHistoryState: this._statusHistoryState,
          statusHistoryNamespace: this._statusHistoryNamespace,
          buttons: this.mode === SharedFileAndAipDetailDialogModeEnum.aip ? [...this._actionsAip, ...this._extraActions] : (this.mode === SharedFileAndAipDetailDialogModeEnum.file ? [...this._actionsFile, ...this._extraActions] : []),
        } as SharedFileAndAipInformationDialogData<TResource>;
        this._changeDetector.detectChanges();
      }),
    );
  }

  protected _retrieveResIdFromUrl(): void {
    this._dataFileId = this._route.snapshot.paramMap.get(AppRoutesEnum.paramIdWithoutPrefixParam);
    this._parentId = this._route.parent.parent.snapshot.paramMap.get(AppRoutesEnum.paramIdWithoutPrefixParam);
  }

  protected _computeMode(): void {
    const url = this._store.selectSnapshot((s: LocalStateModel) => s.router.state.url);
    const route = url.substring(0, url.lastIndexOf(urlSeparator));
    if (route.endsWith("files")) {
      this.mode = SharedFileAndAipDetailDialogModeEnum.file;
    } else if (route.endsWith("collections")) {
      this.mode = SharedFileAndAipDetailDialogModeEnum.aip;
    } else {
      console.error("Unable to define mode file or aip");
    }
  }

  protected _getOrRefreshResource(): void {
    this._store.dispatch(new this._stateAction.GetById(this._parentId, this._dataFileId));
  }

  private _downloadDataFile(): void {
    this._store.dispatch(new this._stateAction["Download"](this._parentId, this.data.dataFile));
  }

  private _resume(): void {
    this._store.dispatch(new this._stateAction["Resume"](this._parentId, this.data.dataFile));
  }

  protected _goToAip(): void {
    this._store.dispatch(new this._stateAction["GoToAip"](this.data.aip));
  }

  delete(): void {
    this.subscribe(this._actions$.pipe(
      ofActionCompleted(this._stateAction.DeleteSuccess),
      filter(result => result.result.successful),
      take(1),
      tap(result => {
        this.sharedFileAndAipInformationContainer.back();
      }),
    ));
    let objectToSend;
    if (this.mode === SharedFileAndAipDetailDialogModeEnum.file) {
      objectToSend = this.data.dataFile;
    } else if (this.mode === SharedFileAndAipDetailDialogModeEnum.aip) {
      objectToSend = this.data.aip;
    }
    this._store.dispatch(new this._stateAction.Delete(this._parentId, objectToSend.resId));
  }
}


