import {
  ChangeDetectorRef,
  Directive,
  OnDestroy,
  OnInit,
  ViewChild,
} from "@angular/core";
import {MatDialog} from "@angular/material/dialog";
import {
  ActivatedRoute,
  RouterStateSnapshot,
} from "@angular/router";
import {SharedAbstractCrudRoutable} from "@app/shared/components/routables/shared-abstract-crud/shared-abstract-crud.routable";
import {StoreDialogUtil} from "@app/shared/utils/store-dialog.util";
import {StoreRouteLocalUtil} from "@app/shared/utils/store-route-local.util";
import {AppBannerAction} from "@app/stores/banner/app-banner.action";
import {AppBannerState} from "@app/stores/banner/app-banner.state";
import {Navigate} from "@ngxs/router-plugin";
import {
  Actions,
  Select,
  Store,
} from "@ngxs/store";
import {SharedAbstractFormPresentational} from "@shared/components/presentationals/shared-abstract-form/shared-abstract-form.presentational";
import {LabelTranslateEnum} from "@shared/enums/label-translate.enum";
import {AppRoutesEnum} from "@shared/enums/routes.enum";
import {CrudHelper} from "@shared/helpers/crud.helper";
import {UrlQueryParamHelper} from "@shared/helpers/url-query-param.helper";
import {FormControlKey} from "@shared/models/form-control-key.model";
import {LocalStateModel} from "@shared/models/local-state.model";
import {Observable} from "rxjs";
import {
  distinctUntilChanged,
  map,
  take,
  tap,
} from "rxjs/operators";
import {
  BaseResourceType,
  isNullOrUndefined,
  MappingObject,
  MemoizedUtil,
  ModelFormControlEvent,
  QueryParameters,
  ResourceActionHelper,
  ResourceNameSpace,
  ResourceStateModel,
  StoreUtil,
} from "solidify-frontend";
import {LocalStateEnum} from "../../../enums/local-state.enum";
import {SharedDeleteDialog} from "../../dialogs/shared-abstract-delete/shared-delete.dialog";

@Directive()
export abstract class SharedAbstractDetailEditRoutable<TResourceModel extends BaseResourceType, UResourceStateModel extends ResourceStateModel<TResourceModel>>
  extends SharedAbstractCrudRoutable<TResourceModel, UResourceStateModel> implements OnInit, OnDestroy {
  @Select((state: LocalStateModel) => state.router.state) urlStateObs: Observable<RouterStateSnapshot>;

  checkAvailableResourceNameSpace: ResourceNameSpace | undefined;
  isLoadingObs: Observable<boolean> = this._store.select(s => StoreUtil.isLoadingState(super.getState(s)));
  queryParametersObs: Observable<QueryParameters> = this._store.select(s => super.getState(s).queryParameters);
  currentObs: Observable<TResourceModel> = this._store.select(s => super.getState(s).current).pipe(
    tap(current => this.current = current),
  );
  current: TResourceModel;

  @ViewChild("formPresentational", {static: true})
  readonly formPresentational: SharedAbstractFormPresentational<any>;

  abstract readonly KEY_PARAM_NAME: keyof TResourceModel & string;

  protected _resId: string;

  isEdit: boolean;
  isEditObs: Observable<boolean>;
  urlQueryParameters: MappingObject<string | undefined>;

  getByIdIfAlreadyInState: boolean = true;
  readonly editAvailable: boolean = true;
  readonly deleteAvailable: boolean = true;

  protected constructor(protected readonly _store: Store,
                        protected readonly _route: ActivatedRoute,
                        protected readonly _actions$: Actions,
                        protected readonly _changeDetector: ChangeDetectorRef,
                        protected readonly _dialog: MatDialog,
                        protected readonly _state: LocalStateEnum,
                        protected readonly _resourceNameSpace: ResourceNameSpace,
                        protected readonly _parentState?: LocalStateEnum) {
    super(_store, _state, _parentState);
  }

  ngOnInit(): void {
    super.ngOnInit();
    this.subscribe(this.getRetrieveEditModeObs());
  }

  ngOnDestroy(): void {
    super.ngOnDestroy();
  }

  protected retrieveCurrentModelAndResource(): void {
    this.retrieveResource();
    this.retrieveCurrentModelWithUrl();
  }

  protected retrieveResource(): void {
    this._store.dispatch(ResourceActionHelper.loadResource(this._resourceNameSpace));
  }

  protected cleanState(): void {
    const queryParameters = this._store.selectSnapshot(s => super.getState(s).queryParameters);
    this._store.dispatch(ResourceActionHelper.clean(this._resourceNameSpace, false, queryParameters));
  }

  protected retrieveResIdFromUrl(): void {
    this._resId = this._route.snapshot.paramMap.get(AppRoutesEnum.paramIdWithoutPrefixParam);
    if (isNullOrUndefined(this._resId)) {
      this._resId = this._route.parent.snapshot.paramMap.get(AppRoutesEnum.paramIdWithoutPrefixParam);
    }
  }

  protected retrieveCurrentModelWithUrl(): void {
    this.retrieveResIdFromUrl();
    if (this.getByIdIfAlreadyInState || isNullOrUndefined(this.current) || this.current?.resId !== this._resId) {
      this.getById(this._resId);
    }
  }

  protected getById(id: string): void {
    this._store.dispatch(ResourceActionHelper.getById(this._resourceNameSpace, id));
    this.getSubResourceWithParentId(id);
  }

  protected abstract getSubResourceWithParentId(id: string): void;

  protected getRetrieveEditModeObs(): Observable<boolean> {
    this.isEditObs = this.urlStateObs.pipe(
      map(url => url.url),
      distinctUntilChanged(),
      map(urlRaw => {
        const urlSplitted = urlRaw.split("?");
        const url = urlSplitted[0];
        this._extractQueryParams(urlSplitted.length > 1 ? urlSplitted[1] : undefined);
        this.isEdit = url.endsWith(AppRoutesEnum.separator + AppRoutesEnum.edit);
        this._changeDetector.detectChanges();
        const bannerStateModel = MemoizedUtil.selectSnapshot(this._store, AppBannerState, state => state);
        if (this.isEdit) {
          this._store.dispatch(new AppBannerAction.ShowInEditMode());
        } else if (bannerStateModel.message === LabelTranslateEnum.youAreInEditMode) {
          this._store.dispatch(new AppBannerAction.Hide());
        }
        return this.isEdit;
      }),
    );
    return this.isEditObs;
  }

  edit(): void {
    if (this.isEdit) {
      return;
    }
    if (!this.editAvailable) {
      return;
    }
    this._store.dispatch(new Navigate([StoreRouteLocalUtil.getEditRoute(this._state, this._resId)], {}, {skipLocationChange: true}));
  }

  update(model: ModelFormControlEvent<TResourceModel>): Observable<any> {
    super.saveInProgress();
    return this._store.dispatch(ResourceActionHelper.update<TResourceModel>(this._resourceNameSpace, model));
  }

  delete(): void {
    const datas = StoreDialogUtil.deleteData(this._state);
    this.subscribe(
      this.currentObs.pipe(take(1)),
      (model: TResourceModel) => {
        datas.name = model[this.KEY_PARAM_NAME]?.toString();
        datas.resId = model.resId;
      });
    this._dialog.open(SharedDeleteDialog, {
      width: "400px",
      data: datas,
    });
  }

  backToList(): void {
    this._store.dispatch(new Navigate([StoreRouteLocalUtil.getRootRoute(this._state)]));
  }

  backToDetail(path: string[] = undefined): void {
    const currentUrl = this._store.selectSnapshot((s: LocalStateModel) => s.router.state.url);
    this.subscribe(this._store.dispatch(new Navigate(isNullOrUndefined(path) ? [StoreRouteLocalUtil.getDetailRoute(this._state), this._resId] : path))
      .pipe(
        tap((state: LocalStateModel) => {
          if (state.router.state.url !== currentUrl) {
            // this.formPresentational.resetFormToInitialValue();
            // TODO : Fix don't need to get by id model from backend but fix resetFormToInitialValue with component user role org unit
            // TODO : Problem currently if redirect to detail page via breadcrumb in case of Deposit
            this.retrieveCurrentModelWithUrl();
          }
        }),
      ));
  }

  checkAvailable(formControlKey: FormControlKey): void {
    const resourceNameSpace = isNullOrUndefined(this.checkAvailableResourceNameSpace) ? this._resourceNameSpace : this.checkAvailableResourceNameSpace;
    this.subscribe(CrudHelper.checkAvailable(formControlKey, this._store, this._actions$, resourceNameSpace, this._resId));
  }

  private _extractQueryParams(rawQueryParameters: string): void {
    this.urlQueryParameters = UrlQueryParamHelper.getQueryParamMappingObjectFromRawQueryParamString(rawQueryParameters);
  }
}
