import {
  Directive,
  ViewChild,
} from "@angular/core";
import {LocalStateEnum} from "@app/shared/enums/local-state.enum";
import {AppAction} from "@app/stores/app.action";
import {Navigate} from "@ngxs/router-plugin";
import {Store} from "@ngxs/store";
import {SharedAbstractFormPresentational} from "@shared/components/presentationals/shared-abstract-form/shared-abstract-form.presentational";
import {SharedAbstractResourceRoutable} from "@shared/components/routables/shared-abstract-resource/shared-abstract-resource.routable";
import {ICanDeactivate} from "@shared/models/can-deactivate.model";
import {
  BaseResourceType,
  isArray,
  isFalse,
  isTrue,
  ResourceStateModel,
} from "solidify-frontend";

@Directive()
export abstract class SharedAbstractCrudRoutable<TResourceModel extends BaseResourceType, UResourceStateModel extends ResourceStateModel<TResourceModel>>
  extends SharedAbstractResourceRoutable<TResourceModel, UResourceStateModel>
  implements ICanDeactivate {
  @ViewChild("formPresentational")
  readonly formPresentational: SharedAbstractFormPresentational<TResourceModel>;

  private _canDeactivate: boolean = true;

  protected constructor(protected readonly _store: Store,
                        protected readonly _state: LocalStateEnum,
                        protected readonly _parentState?: LocalStateEnum) {
    super(_state, _parentState);
  }

  canDeactivate(): boolean {
    return this._canDeactivate;
  }

  updateCanDeactivate(isDirty: boolean): void {
    if (isTrue(isDirty) && isDirty === this._canDeactivate) {
      this.preventExit();
    } else if (isFalse(isDirty) && isDirty === this._canDeactivate) {
      this.cancelPreventExit();
    }
    this._canDeactivate = !isDirty;
  }

  saveInProgress(): void {
    this.cancelPreventExit();
  }

  preventExit(): void {
    this._canDeactivate = false;
    this._store.dispatch(new AppAction.PreventExit());
  }

  cancelPreventExit(): void {
    this._canDeactivate = true;
    this._store.dispatch(new AppAction.CancelPreventExit());
  }

  navigate(path: string | string[]): void {
    this._store.dispatch(new Navigate(isArray(path) ? path : [path]));
  }

  navigateNavigate(navigate: Navigate): void {
    this._store.dispatch(navigate);
  }

  navigateWithQueryParam(navigateAction: Navigate): void {
    this._store.dispatch(navigateAction);
  }
}
