import {
  ChangeDetectorRef,
  Directive,
  OnDestroy,
  OnInit,
} from "@angular/core";
import {SharedAbstractCrudRoutable} from "@app/shared/components/routables/shared-abstract-crud/shared-abstract-crud.routable";
import {Navigate} from "@ngxs/router-plugin";
import {
  Actions,
  Store,
} from "@ngxs/store";
import {CrudHelper} from "@shared/helpers/crud.helper";
import {UrlQueryParamHelper} from "@shared/helpers/url-query-param.helper";
import {FormControlKey} from "@shared/models/form-control-key.model";
import {StoreRouteLocalUtil} from "@shared/utils/store-route-local.util";
import {Observable} from "rxjs";
import {
  BaseResourceType,
  isNullOrUndefined,
  MappingObject,
  ModelFormControlEvent,
  QueryParameters,
  ResourceActionHelper,
  ResourceNameSpace,
  ResourceStateModel,
  StoreUtil,
} from "solidify-frontend";
import {LocalStateEnum} from "../../../enums/local-state.enum";

@Directive()
export abstract class SharedAbstractCreateRoutable<TResourceModel extends BaseResourceType, UResourceStateModel extends ResourceStateModel<TResourceModel>> extends SharedAbstractCrudRoutable<TResourceModel, UResourceStateModel> implements OnInit, OnDestroy {
  isLoadingObs: Observable<boolean>;
  urlQueryParameters: MappingObject<string | undefined>;
  checkAvailableResourceNameSpace: ResourceNameSpace | undefined;

  protected constructor(protected readonly _store: Store,
                        protected readonly _actions$: Actions,
                        protected readonly _changeDetector: ChangeDetectorRef,
                        protected readonly _state: LocalStateEnum,
                        private readonly _resourceActionNameSpace: ResourceNameSpace,
                        protected readonly _parentState?: LocalStateEnum) {
    super(_store, _state, _parentState);
    this.isLoadingObs = this._store.select(s => StoreUtil.isLoadingState(super.getState(s)));
  }

  create(modelFormControlEvent: ModelFormControlEvent<TResourceModel>): void {
    super.saveInProgress();
    this._store.dispatch(ResourceActionHelper.create(this._resourceActionNameSpace, modelFormControlEvent));
  }

  ngOnInit(): void {
    super.ngOnInit();
    this._store.dispatch(ResourceActionHelper.loadResource(this._resourceActionNameSpace));
    this._extractQueryParams();
  }

  ngOnDestroy(): void {
    super.ngOnDestroy();
    this.cleanState();
  }

  protected cleanState(): void {
    this._store.dispatch(ResourceActionHelper.clean(this._resourceActionNameSpace, false, new QueryParameters()));
  }

  checkAvailable(formControlKey: FormControlKey): void {
    const resourceNameSpace = isNullOrUndefined(this.checkAvailableResourceNameSpace) ? this._resourceActionNameSpace : this.checkAvailableResourceNameSpace;
    this.subscribe(CrudHelper.checkAvailable(formControlKey, this._store, this._actions$, resourceNameSpace, undefined));
  }

  backToList(): void {
    this._store.dispatch(new Navigate([StoreRouteLocalUtil.getRootRoute(this._state)]));
  }

  private _extractQueryParams(): void {
    this.urlQueryParameters = UrlQueryParamHelper.getQueryParamMappingObject();
  }
}
