import {DOCUMENT} from "@angular/common";
import {
  AfterViewInit,
  ChangeDetectionStrategy,
  ChangeDetectorRef,
  Component,
  Inject,
  Input,
  OnInit,
  Output,
  Renderer2,
  ViewChild,
} from "@angular/core";
import {
  MatMenu,
  MatMenuItem,
} from "@angular/material/menu";
import {SharedAbstractPresentational} from "@shared/components/presentationals/shared-abstract/shared-abstract.presentational";
import {TocHelper} from "@shared/helpers/toc.helper";
import {BreakpointService} from "@shared/services/breakpoint.service";
import {
  BehaviorSubject,
  Observable,
} from "rxjs";
import {distinctUntilChanged} from "rxjs/operators";
import {
  MARK_AS_TRANSLATABLE,
  ObservableUtil,
} from "solidify-frontend";

@Component({
  selector: "dlcm-shared-toc",
  templateUrl: "./shared-toc.presentational.html",
  styleUrls: ["./shared-toc.presentational.scss"],
  changeDetection: ChangeDetectionStrategy.OnPush,
})
export class SharedTocPresentational extends SharedAbstractPresentational implements AfterViewInit, OnInit {
  @Input()
  quickStart: string;

  @Input()
  integrationGuide: string;

  @Input()
  apis: string;

  @Input()
  toolsGuide: string;

  @Input()
  isLoading: boolean;

  @ViewChild("menu")
  readonly menu: MatMenu;

  @ViewChild("integrationGuideMenu")
  readonly integrationGuideMenu: MatMenu;

  private readonly _displaySidebarUserGuideChangeBS: BehaviorSubject<void> = new BehaviorSubject<void>(undefined);
  @Output("displaySidebarUserGuideChange")
  readonly displaySidebarUserGuideChangeObs: Observable<void> = ObservableUtil.asObservable(this._displaySidebarUserGuideChangeBS);

  matMenuLinkUpdated: MatMenuItem[] = [];
  isMobile: boolean;
  readonly CLASS_MAT_MENU_WITH_LINK: string = "shared-toc-container-panel";

  constructor(@Inject(DOCUMENT) private readonly _document: Document,
              private readonly _renderer: Renderer2,
              public readonly breakpointService: BreakpointService,
              private readonly _changeDetector: ChangeDetectorRef) {
    super();
  }

  ngOnInit(): void {
    this.subscribe(this.breakpointService.isSmallerOrEqualThanMdObs().pipe(
      distinctUntilChanged(),
    ), result => {
      this.isMobile = result;
      this._changeDetector.detectChanges();
    });
  }

  ngAfterViewInit(): void {
    super.ngAfterViewInit();

    this.subscribe(this.menu._hovered(), (matMenuItem: MatMenuItem) => {
      if (this.matMenuLinkUpdated.includes(matMenuItem)) {
        return;
      }
      this.matMenuLinkUpdated.push(matMenuItem);
      setTimeout(() => {
        TocHelper.updateLinkToc(this._document, this._renderer, this.CLASS_MAT_MENU_WITH_LINK, false);
      }, 0);
    });
  }

  getTooltipButton(): string {
    if (this.isMobile) {
      return MARK_AS_TRANSLATABLE("toc.button.doc");
    }
    return MARK_AS_TRANSLATABLE("toc.button.devDoc");
  }

  openUserGuide(): void {
    this._displaySidebarUserGuideChangeBS.next();
  }
}
