import {
  FlexibleConnectedPositionStrategy,
  Overlay,
  OverlayRef,
} from "@angular/cdk/overlay";
import {ComponentPortal} from "@angular/cdk/portal";
import {
  AfterViewInit,
  ChangeDetectionStrategy,
  ChangeDetectorRef,
  Component,
  ComponentRef,
  ElementRef,
  Input,
  NgZone,
  OnDestroy,
  OnInit,
  Output,
  ViewChild,
  ViewContainerRef,
} from "@angular/core";
import {
  ControlValueAccessor,
  FormBuilder,
  FormControl,
  FormGroup,
  NG_VALUE_ACCESSOR,
} from "@angular/forms";
import {FloatLabelType} from "@angular/material/core";
import {MatFormFieldAppearance} from "@angular/material/form-field";
import {SharedAbstractPresentational} from "@app/shared/components/presentationals/shared-abstract/shared-abstract.presentational";
import {environment} from "@environments/environment";
import {TranslateService} from "@ngx-translate/core";
import {
  Actions,
  ofActionCompleted,
  Store,
} from "@ngxs/store";
import {SharedSearchableSingleSelectContentPresentational} from "@shared/components/presentationals/shared-searchable-single-select-content/shared-searchable-single-select-content.presentational";
import {LabelTranslateEnum} from "@shared/enums/label-translate.enum";
import {LocalStorageEnum} from "@shared/enums/local-storage.enum";
import {LocalStorageHelper} from "@shared/helpers/local-storage.helper";
import {BaseFormDefinition} from "@shared/models/base-form-definition.model";
import {TabulationService} from "@shared/services/tabulation.service";
import {isNotNullOrUndefined} from "codelyzer/util/isNotNullOrUndefined";
import {
  BehaviorSubject,
  Observable,
} from "rxjs";
import {
  distinctUntilChanged,
  filter,
  take,
  tap,
} from "rxjs/operators";
import {
  BaseResourceType,
  FormValidationHelper,
  isEmptyString,
  isNonEmptyString,
  isNotNullNorUndefined,
  isNullOrUndefined,
  isTrue,
  MappingObject,
  MemoizedUtil,
  ObservableUtil,
  PropertyName,
  ResourceNameSpace,
  ResourceState,
  ResourceStateModel,
  Sort,
  StringUtil,
  Type,
} from "solidify-frontend";

@Component({
  selector: "dlcm-shared-searchable-single-select",
  templateUrl: "./shared-searchable-single-select.presentational.html",
  styleUrls: ["./shared-searchable-single-select.presentational.scss"],
  changeDetection: ChangeDetectionStrategy.OnPush,
  providers: [
    {
      provide: NG_VALUE_ACCESSOR,
      multi: true,
      useExisting: SharedSearchableSingleSelectPresentational,
    },
  ],
})
export class SharedSearchableSingleSelectPresentational<TStateModel extends ResourceStateModel<TResource>, TResource extends BaseResourceType> extends SharedAbstractPresentational implements ControlValueAccessor, OnInit, AfterViewInit, OnDestroy {
  formDefinition: FormComponentFormDefinition = new FormComponentFormDefinition();
  formLabel: FormGroup;

  protected _overlayRefs: Set<OverlayRef> = new Set<OverlayRef>();

  @Input()
  placeholder: string;

  @Input()
  valueKey: string;

  @Input()
  labelKey: string;

  @Input()
  appearance: MatFormFieldAppearance = this.appearanceInputMaterial;

  _labelCallback: (value: TResource) => string = (value => value[this.labelKey]);

  @Input()
  set labelCallback(value: (value: TResource) => string) {
    this._labelCallback = value;
  }

  get labelCallback(): (value: TResource) => string {
    if (isNullOrUndefined(this._labelCallback)) {
      return (value => value[this.labelKey]);
    }
    return this._labelCallback;
  }

  @Input()
  formControl: FormControl;

  @Input()
  required: boolean;

  @Input()
  canDeleteValue: boolean = true;

  @Input()
  resourceNameSpace: ResourceNameSpace;

  @Input()
  state: Type<ResourceState<TStateModel, TResource>>;

  @Input()
  urlKey: string;

  @Input()
  isWithLink: boolean = false;

  @Input()
  tooltipNavigateToTranslate: string;

  @Input()
  searchKey: string;

  @Input()
  extraSearchQueryParam: MappingObject;

  @Input()
  defaultValue: string | undefined;

  @Input()
  sort: Sort | undefined;

  classMatFormField: string;

  private _isDisabled: boolean = false;

  @Input()
  set isDisabled(value: boolean) {
    this._isDisabled = value;
    if (this._isDisabled) {
      this.classMatFormField = this.classInputIgnored;
    } else {
      this.classMatFormField = StringUtil.stringEmpty;
    }
    this.updateFormReadonlyState();
  }

  get isDisabled(): boolean {
    return this._isDisabled;
  }

  get isOverlayOpen(): boolean {
    return this._overlayRefs?.size > 0;
  }

  @Input()
  extraGetAllParameters: any[] = [];

  @Input()
  storeLastSelectionKey: LocalStorageEnum | undefined = undefined;

  @Input()
  displayCommonValuesListKey: string[] = [];

  @Input()
  displayCommonValuesLabel: string = LabelTranslateEnum.mostCommonValues;

  isDefaultValueSelected: boolean = false;

  private url: string;

  private _readonly: boolean;
  @Input()
  set readonly(value: boolean) {
    this._readonly = value;
    this.updateFormReadonlyState();
  }

  get readonly(): boolean {
    return this._readonly || this.isDisabled;
  }

  currentObs: Observable<TResource>;
  isLoadingObs: Observable<boolean>;

  floatLabel: FloatLabelType;

  alreadyInit: boolean = false;
  noFormControl: boolean = false;

  labelInitialized: boolean = false;

  classInputHide: string = environment.classInputHide;
  classInputIgnored: string = environment.classInputIgnored;

  MAX_ITEM_TO_DISPLAY_IN_LAST_SELECTION: number = 3;

  @ViewChild("inputElementRef")
  readonly input: ElementRef;

  get formValidationHelper(): typeof FormValidationHelper {
    return FormValidationHelper;
  }

  private readonly _valueBS: BehaviorSubject<TResource | undefined> = new BehaviorSubject<TResource | undefined>(undefined);
  @Output("valueChange")
  readonly valueObs: Observable<TResource | undefined> = ObservableUtil.asObservable(this._valueBS);

  protected readonly _navigateBS: BehaviorSubject<string | undefined> = new BehaviorSubject<string | undefined>(undefined);
  @Output("navigate")
  readonly navigateObs: Observable<string | undefined> = ObservableUtil.asObservable(this._navigateBS);

  lastActiveElement: Element;
  objectNotFound: boolean = false;

  constructor(private _store: Store,
              protected _fb: FormBuilder,
              private _elementRef: ElementRef,
              private _viewContainerRef: ViewContainerRef,
              private _overlay: Overlay,
              private _ngZone: NgZone,
              private _tabulationService: TabulationService,
              private _actions$: Actions,
              private _changeDetector: ChangeDetectorRef,
              private _translate: TranslateService) {
    super();
    this._ngZone.runOutsideAngular(() => document.addEventListener("focusin", focusEvent => this.lastActiveElement = document.activeElement));
  }

  ngOnDestroy(): void {
    this.closeOverlays(false, false);
    super.ngOnDestroy();
  }

  ngOnInit(): void {
    if (isNullOrUndefined(this.formControl)) {
      this.noFormControl = true;
      this.formControl = this._fb.control(undefined);
    }
    if (isTrue(this.alreadyInit)) {
      // When validator is updated, the ngOnInit is call a new time
      // This hack prevent this to happen (case with deposit license validator impacted by access level
      return;
    }
    this._initFormLabel();
    this._setDefaultValue();
    this._dispatchGetActionToRetrieveLabel();
    this._observeLabelAndUrlUpdate();
    this.alreadyInit = true;
  }

  private _initFormLabel(): void {
    this.formLabel = this._fb.group({
      [this.formDefinition.value]: [undefined],
    });

    this.updateFormReadonlyState();
  }

  private updateFormReadonlyState(): void {
    if (isNullOrUndefined(this.formLabel)) {
      return;
    }
    if (this.readonly) {
      this.formLabel.disable();
    } else {
      this.formLabel.enable();
    }
  }

  private _setDefaultValue(): void {
    if (isNullOrUndefined(this.defaultValue) || (isNotNullOrUndefined(this.formControl.value) && isNonEmptyString(this.formControl.value))) {
      return;
    }
    this.formControl.setValue(this.defaultValue);
    this.isDefaultValueSelected = true;
  }

  private _dispatchGetActionToRetrieveLabel(): void {
    const key = this.formControl.value;
    this.currentObs = MemoizedUtil.current(this._store, this.state);
    this.isLoadingObs = MemoizedUtil.isLoading(this._store, this.state);
    if (!isNullOrUndefined(key) && isNonEmptyString(key)) {
      this.subscribe(this._actions$.pipe(
        ofActionCompleted(this.resourceNameSpace.GetByIdFail),
        take(1),
        tap((result) => {
          if (isTrue(result.result.successful)) {
            this.objectNotFound = true;
            this.formLabel.get(this.formDefinition.value)
              .setValue(this._translate.instant(LabelTranslateEnum.objectNotFound));
            this._changeDetector.detectChanges();
          }
        }),
      ));

      this._store.dispatch(new this.resourceNameSpace.GetById(key));
    } else {
      this.labelInitialized = true;
    }
  }

  private _observeLabelAndUrlUpdate(): void {
    this.subscribe(this.currentObs.pipe(
      distinctUntilChanged(),
      filter((value) => !isNullOrUndefined(value) && value[this.valueKey] === this.formControl.value),
      take(1),
      tap((value) => {
        if (isNullOrUndefined(this.formControl.value) || isEmptyString(this.formControl.value)) {
          return;
        }
        this.formLabel.get(this.formDefinition.value).setValue(this.labelCallback(value));
        if (!isNullOrUndefined(value[this.urlKey])) {
          this.url = value[this.urlKey];
        }
        this.labelInitialized = true;
        this.computeFloatLabel();
      }),
    ));
  }

  propagateChange = (__: any) => {};

  registerOnChange(fn: any): void {
    this.propagateChange = fn;
  }

  registerOnTouched(fn: any): void {
  }

  setDisabledState(isDisabled: boolean): void {
  }

  writeValue(value: string): void {
    // if (!isNullOrUndefined(value) && !this.noFormControl) {
    //   this.formControl.setValue(value);
    //   this._dispatchGetActionToRetrieveLabel();
    // }
  }

  get openElementToRefocus(): Element {
    return this._openElementToRefocus;
  }

  private _openElementToRefocus: Element;

  set openElementToRefocus(element: Element) {
    this._openElementToRefocus = element;
  }

  @Input()
  defaultValueTooltip: string | undefined;

  openOverlay(focusEvent: FocusEvent): void {
    this.openElementToRefocus = this.lastActiveElement;
    const widthField = this._elementRef.nativeElement.getBoundingClientRect().width;
    const overlayRef = this._overlay.create({
      positionStrategy: this._getPosition(this._elementRef),
      scrollStrategy: this._overlay.scrollStrategies.block(),
      hasBackdrop: true,
      backdropClass: "cdk-overlay-transparent-backdrop",
      width: widthField,
    });

    this._overlayRefs.add(overlayRef);

    const overlayContentComponent = new ComponentPortal<SharedSearchableSingleSelectContentPresentational<TResource>>(SharedSearchableSingleSelectContentPresentational);
    const componentRef = overlayRef.attach(overlayContentComponent);
    componentRef.instance.host = this;

    this._observeValueSelectedInOverlay(componentRef, overlayRef);
    this._observeCloseEventInOverlay(componentRef, overlayRef);
    this._observeClickBackdrop(componentRef, overlayRef);
    this._simulateClickEvent();
  }

  private _simulateClickEvent(): void {
    // For enter in edit mode
    this.input.nativeElement.dispatchEvent(new MouseEvent("click", {
      bubbles: true,
      cancelable: true,
      view: window,
    }));
  }

  private _observeValueSelectedInOverlay(componentRef: ComponentRef<SharedSearchableSingleSelectContentPresentational<TResource>>, overlayRef: OverlayRef): void {
    this.subscribe(componentRef.instance.valueObs.pipe(
      distinctUntilChanged(),
      tap(value => {
        const valueKey = value[this.valueKey];

        if (isNotNullNorUndefined(this.storeLastSelectionKey)) {
          LocalStorageHelper.addItemInList(this.storeLastSelectionKey, valueKey, this.MAX_ITEM_TO_DISPLAY_IN_LAST_SELECTION + 1);
        }
        this.propagateChange(valueKey);
        this.formControl.setValue(valueKey);
        this._valueBS.next(value);
        this.objectNotFound = false;
        if (!isNullOrUndefined(value[this.urlKey])) {
          this.url = value[this.urlKey];
        }
        this.formLabel.get(this.formDefinition.value).setValue(this.labelCallback(value));
        this.computeFloatLabel();
        this._closeOverlay(overlayRef);
      }),
    ));
  }

  closeOverlays(isFocusNext: boolean = true, isFocusedByTab: boolean = false): void {
    this._overlayRefs.forEach(overlayRef => this._closeOverlay(overlayRef, true));
    this._overlayRefs.clear();
    if (isFocusNext) {
      const elementToFocus = isFocusedByTab ? this._tabulationService.getByDelta(2, this.openElementToRefocus) : this._tabulationService.getNext(this.openElementToRefocus);
      this._tabulationService.focusNext(elementToFocus);
    } else {
      const elementToFocus = isFocusedByTab ? this._tabulationService.getByDelta(-2, this.openElementToRefocus) : this._tabulationService.getPrev(this.openElementToRefocus);
      this._tabulationService.focusPrev(elementToFocus);
    }
  }

  private _observeCloseEventInOverlay(componentRef: ComponentRef<SharedSearchableSingleSelectContentPresentational<TResource>>, overlayRef: OverlayRef): void {
    this.subscribe(componentRef.instance.closeObs.pipe(
      tap(() => {
        this._closeOverlay(overlayRef);
      }),
    ));
  }

  private _observeClickBackdrop(componentRef: ComponentRef<SharedSearchableSingleSelectContentPresentational<TResource>>, overlayRef: OverlayRef): void {
    this.subscribe(overlayRef.backdropClick().pipe(
      tap(() => {
        this._closeOverlay(overlayRef);
      }),
    ));
    this.subscribe(componentRef.instance.closeByTabObs.pipe(
      tap(() => {
        this.closeOverlays();
      }),
    ));
  }

  private _closeOverlay(overlayRef: OverlayRef, keepInOverlayRefs?: boolean): void {
    overlayRef.detach();
    if (!keepInOverlayRefs) {
      this._overlayRefs.delete(overlayRef);
    }
    this.formControl.markAsTouched();
    this._changeDetector.detectChanges();
  }

  private _getPosition(elementToConnectTo: ElementRef): FlexibleConnectedPositionStrategy {
    return this._overlay.position()
      .flexibleConnectedTo(elementToConnectTo)
      .withFlexibleDimensions(false)
      .withPositions([
        {
          originX: "start",
          originY: "top",
          overlayX: "start",
          overlayY: "top",
        },
      ]);
  }

  isValuePresent(): boolean {
    return isNonEmptyString(this.formControl.value);
  }

  isUrlPresent(): boolean {
    return this.isValuePresent() && !isNullOrUndefined(this.url) && isNonEmptyString(this.url);
  }

  clearValue($event: MouseEvent): void {
    if (this.defaultValue) {
      this.formControl.setValue(this.defaultValue);
      this.labelInitialized = false;
      this._dispatchGetActionToRetrieveLabel();
      this._observeLabelAndUrlUpdate();
      this.isDefaultValueSelected = true;
    } else {
      this.formControl.setValue("");
      this.formLabel.get(this.formDefinition.value).setValue(null);
      this._valueBS.next(undefined);
      this.computeFloatLabel();
    }
    $event.stopPropagation();
  }

  navigateTo(event: Event): void {
    if (isNotNullNorUndefined(event)) {
      event.stopPropagation();
      event.preventDefault();
    }
    this._navigateBS.next(this.formControl.value);
  }

  navigateToUrl($event?: MouseEvent): void {
    window.open(this.url, "_blank");
    if (!isNullOrUndefined($event)) {
      $event.stopPropagation();
    }
  }

  computeFloatLabel(): void {
    const value = this.formLabel.get(this.formDefinition.value).value;
    if (isNullOrUndefined(value) || isEmptyString(value)) {
      this.floatLabel = "never";
    } else {
      this.floatLabel = "always";
    }
    this._changeDetector.detectChanges();
  }

  focusInputIfNotClickTarget($event: MouseEvent): void {
    if (!isNullOrUndefined(this.input) && $event.target !== this.input.nativeElement) {
      this.input.nativeElement.focus();
    }
  }
}

class FormComponentFormDefinition extends BaseFormDefinition {
  @PropertyName() search: string;
  @PropertyName() value: string;
}
