import {Overlay} from "@angular/cdk/overlay";
import {ComponentPortal} from "@angular/cdk/portal";
import {
  ChangeDetectionStrategy,
  ChangeDetectorRef,
  Component,
  ElementRef,
  Input,
  OnInit,
  ViewChild,
  ViewContainerRef,
} from "@angular/core";
import {
  ControlValueAccessor,
  FormBuilder,
  NG_VALUE_ACCESSOR,
} from "@angular/forms";
import {MatFormFieldAppearance} from "@angular/material/form-field";
import {environment} from "@environments/environment";
import {Store} from "@ngxs/store";
import {SharedAbstractMultiSelectPresentational} from "@shared/components/presentationals/shared-abstract-multi-select/shared-abstract-multi-select.presentational";
import {SharedSearchableMultiSelectContentPresentational} from "@shared/components/presentationals/shared-searchable-multi-select-content/shared-searchable-multi-select-content.presentational";
import {LabelTranslateEnum} from "@shared/enums/label-translate.enum";
import {LocalStorageEnum} from "@shared/enums/local-storage.enum";
import {LocalStorageHelper} from "@shared/helpers/local-storage.helper";
import {BaseFormDefinition} from "@shared/models/base-form-definition.model";
import {TabulationService} from "@shared/services/tabulation.service";
import {Observable} from "rxjs";
import {
  distinctUntilChanged,
  filter,
  takeWhile,
  tap,
} from "rxjs/operators";
import {
  BaseResourceType,
  isEmptyArray,
  isNonEmptyArray,
  isNotNullNorUndefined,
  isNullOrUndefined,
  MappingObject,
  MemoizedUtil,
  NotificationService,
  PropertyName,
  ResourceNameSpace,
  ResourceState,
  ResourceStateModel,
  Sort,
  StringUtil,
  Type,
} from "solidify-frontend";

@Component({
  selector: "dlcm-shared-searchable-multi-select",
  templateUrl: "./shared-searchable-multi-select.presentational.html",
  styleUrls: ["./shared-searchable-multi-select.presentational.scss"],
  changeDetection: ChangeDetectionStrategy.OnPush,
  providers: [
    {
      provide: NG_VALUE_ACCESSOR,
      multi: true,
      useExisting: SharedSearchableMultiSelectPresentational,
    },
  ],
})
export class SharedSearchableMultiSelectPresentational<TStateModel extends ResourceStateModel<TResource>, TResource extends BaseResourceType> extends SharedAbstractMultiSelectPresentational<TResource> implements ControlValueAccessor, OnInit {
  @Input()
  resourceNameSpace: ResourceNameSpace;

  @Input()
  state: Type<ResourceState<TStateModel, TResource>>;

  @Input()
  extraSearchQueryParam: MappingObject;

  @Input()
  searchKey: string;

  @Input()
  sort: Sort | undefined;

  @Input()
  extraGetAllParameters: any[] = [];

  @ViewChild("inputElementRef", {static: true})
  inputElementRef: ElementRef;

  @Input()
  storeLastSelectionKey: LocalStorageEnum | undefined = undefined;

  @Input()
  appearance: MatFormFieldAppearance = this.appearanceInputMaterial;

  classMatFormField: string;
  MAX_ITEM_TO_DISPLAY_IN_LAST_SELECTION: number = 3;

  private _isDisabled: boolean = false;

  @Input()
  set isDisabled(value: boolean) {
    this._isDisabled = value;
    if (this._isDisabled) {
      this.classMatFormField = this.classInputIgnored;
    } else {
      this.classMatFormField = StringUtil.stringEmpty;
    }
  }

  get isDisabled(): boolean {
    return this._isDisabled;
  }

  @Input()
  displayCommonValuesListKey: string[] = [];

  @Input()
  displayCommonValuesLabel: string = LabelTranslateEnum.mostCommonValues;

  listObs: Observable<TResource[]>;
  isLoadingObs: Observable<boolean>;

  labelInitialized: boolean = false;
  isInitialized: boolean = false;

  classInputIgnored: string = environment.classInputIgnored;

  constructor(protected _fb: FormBuilder,
              protected _elementRef: ElementRef,
              protected _viewContainerRef: ViewContainerRef,
              protected _overlay: Overlay,
              protected _changeDetector: ChangeDetectorRef,
              protected _notificationService: NotificationService,
              protected _tabulationService: TabulationService,
              private _store: Store) {
    super(_fb,
      _elementRef,
      _viewContainerRef,
      _overlay,
      _changeDetector,
      _notificationService,
      _tabulationService);
  }

  getComponentPortal(): ComponentPortal<SharedSearchableMultiSelectContentPresentational<TResource>> {
    return new ComponentPortal<SharedSearchableMultiSelectContentPresentational<TResource>>(SharedSearchableMultiSelectContentPresentational);
  }

  ngOnInit(): void {
    if (this.isInitialized) {
      return;
    }
    super.ngOnInit();
    // this._managePostExternalChange();
    this.isInitialized = true;
    if (this.isDisabled) {
      this.isDisabled = true;
    }
  }

  private _managePostExternalChange(): void {
    this.subscribe(this.formControl.valueChanges.pipe(
      distinctUntilChanged(),
      tap((values: string[]) => {
        values.forEach(v => {
          if (!isNullOrUndefined(this.searchItemInSelectedItemsWithId(v))) {
            return;
          }
          // TODO manage here external add with id (need to retrieve full object in backend)
        });
      }),
    ));
  }

  protected _manageInitExistingValue(): void {
    const keys = this.formControl.value;
    if (isNonEmptyArray(keys)) {
      this._dispatchGetActionToRetrieveLabel(keys);
      this._getLabelInitialSelectedValue();
    } else {
      this.labelInitialized = true;
    }
  }

  private _dispatchGetActionToRetrieveLabel(listResId: string[]): void {
    this.listObs = MemoizedUtil.list(this._store, this.state);
    this.isLoadingObs = MemoizedUtil.isLoading(this._store, this.state);
    this._store.dispatch(new this.resourceNameSpace.GetByListId(listResId));
  }

  private _getLabelInitialSelectedValue(): void {
    this.subscribe(this.listObs.pipe(
      distinctUntilChanged(),
      filter((list: TResource[]) => !isNullOrUndefined(list) && !isEmptyArray(list)),
      takeWhile((list: TResource[]) => this.selectedItems.length < this.formControl.value.length),
      tap((list) => {
        const keys = this.formControl.value;
        keys.forEach(key => {
          const elementAlreadyPresent = this.selectedItems.find(i => i[this.valueKey] === key);
          if (!isNullOrUndefined(elementAlreadyPresent)) {
            return;
          }
          const element = list.find(i => i[this.valueKey] === key);
          if (isNullOrUndefined(element)) {
            // Element with key ${key} not found in list for searchable-multi-select '${this.title}'. Wait next list update
            return;
          }
          this.selectedItems.push(element);
        });

        if (this.selectedItems.length === this.formControl.value.length) {
          this.labelInitialized = true;
          this._changeDetector.detectChanges();
        }
      }),
    ));
  }

  protected extraBehaviorOnValueSelectedInOvertly(value: TResource): void {
    if (isNotNullNorUndefined(this.storeLastSelectionKey)) {
      LocalStorageHelper.addItemInList(this.storeLastSelectionKey, value[this.valueKey], this.MAX_ITEM_TO_DISPLAY_IN_LAST_SELECTION + 1);
    }
  }
}

class FormComponentFormDefinition extends BaseFormDefinition {
  @PropertyName() search: string;
  @PropertyName() value: string;
}
