import {OverlayRef} from "@angular/cdk/overlay";
import {
  AfterViewInit,
  ChangeDetectorRef,
  Directive,
  ElementRef,
  HostBinding,
  HostListener,
  Input,
  OnInit,
  ViewChild,
} from "@angular/core";
import {
  FormBuilder,
  FormGroup,
} from "@angular/forms";
import {environment} from "@environments/environment";
import {
  Actions,
  ofActionCompleted,
  Store,
} from "@ngxs/store";
import {SharedAbstractContentPresentational} from "@shared/components/presentationals/shared-abstract-content/shared-abstract-content.presentational";
import {SharedSearchableMultiSelectPresentational} from "@shared/components/presentationals/shared-searchable-multi-select/shared-searchable-multi-select.presentational";
import {SharedSearchableSingleSelectPresentational} from "@shared/components/presentationals/shared-searchable-single-select/shared-searchable-single-select.presentational";
import {LocalStorageHelper} from "@shared/helpers/local-storage.helper";
import {Observable} from "rxjs";
import {
  debounceTime,
  distinctUntilChanged,
  filter,
  take,
  tap,
} from "rxjs/operators";
import {
  BaseResourceType,
  isArray,
  isEmptyArray,
  isEmptyString,
  isFalse,
  isNonEmptyString,
  isNotNullNorUndefined,
  isNullOrUndefined,
  MappingObjectUtil,
  MemoizedUtil,
  QueryParameters,
  ResourceAction,
  ResourceStateModel,
} from "solidify-frontend";

@Directive()
export abstract class SharedSearchableAbstractContentPresentational<TResource extends BaseResourceType> extends SharedAbstractContentPresentational<TResource> implements OnInit, AfterViewInit {
  listObs: Observable<TResource[]>;
  totalObs: Observable<number>;
  isLoadingObs: Observable<boolean>;
  isLoadingChunkObs: Observable<boolean>;
  navigationResId: string;

  form: FormGroup;

  protected _overlayRefs: Set<OverlayRef> = new Set<OverlayRef>();

  @HostBinding("class.is-loading")
  isLoading: boolean;

  list: TResource[];

  listLastSelection: TResource[] = [];

  listCommonValues: TResource[] = [];

  @Input()
  abstract host: SharedSearchableMultiSelectPresentational<ResourceStateModel<TResource>, TResource> | SharedSearchableSingleSelectPresentational<ResourceStateModel<TResource>, TResource>;

  @ViewChild("input")
  input: ElementRef;

  @HostListener("scroll", ["$event"])
  onWindowScroll(): void {
    if (MemoizedUtil.totalSnapshot(this._store, this.host.state) === MemoizedUtil.listSnapshot(this._store, this.host.state).length) {
      return;
    }
    const currentPosition = this._elementRef.nativeElement.scrollTop + this._elementRef.nativeElement.offsetHeight;
    const maxPosition = this._elementRef.nativeElement.scrollHeight - 1;

    if (currentPosition >= maxPosition) {
      this.loadNextChunk();
    }
  }

  protected readonly _debounceTime: number = 250;
  protected readonly _pageSize: number = environment.defaultEnumValuePageSizeLazyLoad;

  constructor(protected readonly _store: Store,
              protected readonly _fb: FormBuilder,
              protected readonly _elementRef: ElementRef,
              protected readonly _changeDetector: ChangeDetectorRef,
              protected readonly _actions$: Actions) {
    super();
  }

  ngOnInit(): void {
    super.ngOnInit();

    this._initSearchForm();
    this._dispatchInitDataLoad();
    this._retrieveSelector();
    this._observeSearchInput();
    this._observeIsLoading();
    this._observeList();
    this._manageCommonValue();
    this._manageLastSelection();
  }

  sortLastResultAlphabetically(a: TResource, b: TResource, key: string): number {
    if (a[key] < b[key]) {
      return -1;
    }
    if (a[key] > b[key]) {
      return 1;
    }
    return 0;
  }

  isActiveSearch(): boolean {
    return isNonEmptyString(this.form.get(this.host.formDefinition.search).value);
  }

  private _initSearchForm(): void {
    this.form = this._fb.group({
      [this.host.formDefinition.search]: [""],
    });
  }

  getFocusedResourceAndIndex(): { value: TResource | undefined, index: number } {
    const list = this.list;
    for (let index = 0; index < list.length; index++) {
      if (this.navigationResId === list[index].resId) {
        return {value: list[index], index};
      }
    }
    return {value: undefined, index: -1};
  }

  getFocusedResource(): TResource | undefined {
    return this.getFocusedResourceAndIndex().value;
  }

  getFocusedIndex(): number {
    return this.getFocusedResourceAndIndex().index;
  }

  selectNextResource(): void {
    const list = this.list;
    const focusedIndex = this.getFocusedIndex();
    if (focusedIndex !== list.length - 1) {
      this.navigationResId = list[focusedIndex + 1].resId;
    }
  }

  selectPreviousResource(): void {
    const list = this.list;
    const focusedIndex = this.getFocusedIndex();
    if (focusedIndex > 0) {
      this.navigationResId = list[focusedIndex - 1].resId;
    } else {
      this.navigationResId = undefined;
    }
  }

  @HostListener("keydown.arrowDown", ["$event"])
  onArrowDown(keyboardEvent: KeyboardEvent): void {
    if (keyboardEvent && !keyboardEvent.defaultPrevented) {
      keyboardEvent.preventDefault();
      this.selectNextResource();
    }
  }

  @HostListener("keydown.arrowUp", ["$event"])
  onArrowUp(keyboardEvent: KeyboardEvent): void {
    if (keyboardEvent && !keyboardEvent.defaultPrevented) {
      keyboardEvent.preventDefault();
      this.selectPreviousResource();
    }
  }

  @HostListener("keydown.enter", ["$event"])
  onSelectValue(keyboardEvent: KeyboardEvent): void {
    if (keyboardEvent && !keyboardEvent.defaultPrevented) {
      keyboardEvent.preventDefault();
      this.select(this.getFocusedResource());
    }
  }

  private _dispatchInitDataLoad(): void {
    const queryParameters = new QueryParameters(this._pageSize, this.host.sort);
    if (isNotNullNorUndefined(this.host.extraSearchQueryParam)) {
      queryParameters.search.searchItems = MappingObjectUtil.copy(this.host.extraSearchQueryParam);
    }
    this._store.dispatch(new this.host.resourceNameSpace.GetAll(queryParameters, false, true, ...this.host.extraGetAllParameters));
  }

  private _retrieveSelector(): void {
    this.listObs = MemoizedUtil.list(this._store, this.host.state);
    this.totalObs = MemoizedUtil.total(this._store, this.host.state);
    this.isLoadingObs = MemoizedUtil.isLoading(this._store, this.host.state);
    this.isLoadingChunkObs = MemoizedUtil.select(this._store, this.host.state, state => state.isLoadingChunk);
  }

  private _observeSearchInput(): void {
    this.subscribe(this.form.get(this.host.formDefinition.search).valueChanges.pipe(
      debounceTime(this._debounceTime),
      distinctUntilChanged(),
      tap(v => {
        const queryParameters = new QueryParameters(this._pageSize, this.host.sort);
        if (isNotNullNorUndefined(this.host.extraSearchQueryParam)) {
          queryParameters.search.searchItems = MappingObjectUtil.copy(this.host.extraSearchQueryParam);
        }
        MappingObjectUtil.set(queryParameters.search.searchItems, (isNotNullNorUndefined(this.host.searchKey) ? this.host.searchKey : this.host.labelKey), v);
        this._store.dispatch(new this.host.resourceNameSpace.GetAll(queryParameters, true, true, ...this.host.extraGetAllParameters));
      }),
    ));
  }

  ngAfterViewInit(): void {
    this.setFocusOnInput();
  }

  private setFocusOnInput(): void {
    setTimeout(() => {
      this.input.nativeElement.focus();
    }, 0);
  }

  private _observeIsLoading(): void {
    this.subscribe(this.isLoadingObs.pipe(
      distinctUntilChanged(),
      tap(isLoading => this.isLoading = isLoading),
    ));
  }

  private _observeList(): void {
    this.subscribe(this.listObs.pipe(
      distinctUntilChanged(),
      tap(list => this.list = list),
    ));
  }

  computeHighlightText(valueElement: string): string {
    const fc = this.form.get(this.host.formDefinition.search);
    if (isNullOrUndefined(fc)) {
      return valueElement;
    }
    const value = fc.value;
    if (isNullOrUndefined(value) || isEmptyString(value)) {
      return valueElement;
    }
    return valueElement.replace(new RegExp(value, "gi"), match => `<span class="highlight-text">${match}</span>`);
  }

  loadNextChunk(): void {
    this._store.dispatch(new this.host.resourceNameSpace.LoadNextChunkList());
  }

  private _manageCommonValue(): void {
    if (isNullOrUndefined(this.host.displayCommonValuesListKey)) {
      return;
    }

    const listCommonValuesId = [...this.host.displayCommonValuesListKey];
    listCommonValuesId.forEach(id => {
      this.subscribe(this._actions$.pipe(
        ofActionCompleted(this.host.resourceNameSpace.GetByIdSuccess),
        filter(result => {
          const action: ResourceAction.GetByIdSuccess<TResource> = result.action;
          return id === action.model.resId;
        }),
        take(1),
        tap((result) => {
          if (isFalse(result.result.successful)) {
            return;
          }
          const action: ResourceAction.GetByIdSuccess<TResource> = result.action;
          this.listCommonValues.push(action.model);
          this.listCommonValues.sort((a, b) => this.sortLastResultAlphabetically(a, b, this.host.labelKey));
          this._changeDetector.detectChanges();
        }),
      ));

      this._store.dispatch(new this.host.resourceNameSpace.GetById(id));
    });
  }

  private _manageLastSelection(): void {
    if (isNullOrUndefined(this.host.storeLastSelectionKey)) {
      return;
    }

    let listLastSelectionId = LocalStorageHelper.getListItem(this.host.storeLastSelectionKey);
    listLastSelectionId = this.reworkListLastSelectedValues(listLastSelectionId);
    listLastSelectionId.forEach(id => {
      this.subscribe(this._actions$.pipe(
        ofActionCompleted(this.host.resourceNameSpace.GetByIdSuccess),
        filter(result => {
          const action: ResourceAction.GetByIdSuccess<TResource> = result.action;
          return id === action.model.resId;
        }),
        take(1),
        tap((result) => {
          if (isFalse(result.result.successful)) {
            return;
          }
          const action: ResourceAction.GetByIdSuccess<TResource> = result.action;
          this.listLastSelection.push(action.model);
          this.listLastSelection.sort((a, b) => this.sortLastResultAlphabetically(a, b, this.host.labelKey));
          this._changeDetector.detectChanges();
        }),
      ));

      this._store.dispatch(new this.host.resourceNameSpace.GetById(id));
    });
  }

  private reworkListLastSelectedValues(listLastSelectionId: string[]): string[] {
    listLastSelectionId = this.removeCurrentValueInLastSelectedValues(listLastSelectionId);
    listLastSelectionId = this.removeSuperfluousLastSelectedValues(listLastSelectionId);
    return listLastSelectionId;
  }

  private removeCurrentValueInLastSelectedValues(listLastSelectionId: string[]): string[] {
    const currentSelectedId = this.host.formControl.value;
    if (isNullOrUndefined(currentSelectedId) || isEmptyString(currentSelectedId) || isEmptyArray(currentSelectedId)) {
      return listLastSelectionId;
    }
    let listCurrentSelectedId = [];
    if (isArray(currentSelectedId)) {
      listCurrentSelectedId = currentSelectedId;
    } else {
      listCurrentSelectedId = [currentSelectedId];
    }
    listCurrentSelectedId.forEach(id => {
      const indexOf = listLastSelectionId.indexOf(id);
      if (indexOf !== -1) {
        listLastSelectionId.splice(indexOf, 1);
      }
    });
    return listLastSelectionId;
  }

  private removeSuperfluousLastSelectedValues(listLastSelectionId: string[]): string[] {
    if (listLastSelectionId.length > this.host.MAX_ITEM_TO_DISPLAY_IN_LAST_SELECTION) {
      listLastSelectionId = listLastSelectionId.splice(listLastSelectionId.length - this.host.MAX_ITEM_TO_DISPLAY_IN_LAST_SELECTION, listLastSelectionId.length);
    }
    return listLastSelectionId;
  }
}



