import {
  ChangeDetectionStrategy,
  Component,
  Input,
  OnInit,
  Output,
} from "@angular/core";
import {
  FormBuilder,
  FormControl,
  FormGroup,
} from "@angular/forms";
import {MatFormFieldAppearance} from "@angular/material/form-field";
import {SharedAbstractPresentational} from "@app/shared/components/presentationals/shared-abstract/shared-abstract.presentational";
import {IconNameEnum} from "@shared/enums/icon-name.enum";
import {LabelTranslateEnum} from "@shared/enums/label-translate.enum";
import {BaseFormDefinition} from "@shared/models/base-form-definition.model";
import {
  BehaviorSubject,
  Observable,
} from "rxjs";
import {
  debounceTime,
  distinctUntilChanged,
  tap,
} from "rxjs/operators";
import {
  isNonEmptyString,
  isNullOrUndefined,
  ObservableUtil,
  PropertyName,
  StringUtil,
} from "solidify-frontend";

@Component({
  selector: "dlcm-shared-search",
  templateUrl: "./shared-search.presentational.html",
  styleUrls: ["./shared-search.presentational.scss"],
  changeDetection: ChangeDetectionStrategy.OnPush,
})
export class SharedSearchPresentational extends SharedAbstractPresentational implements OnInit {
  formDefinition: FormComponentFormDefinition = new FormComponentFormDefinition();

  searchForm: FormGroup;

  readonly MODE_VALIDATION: string = "validation";
  readonly MODE_AUTO: string = "auto";

  private _value: string;

  @Input()
  appearance: MatFormFieldAppearance = "legacy";

  @Input()
  searchAutoFocus: boolean = false;

  @Input()
  placeholder: string | undefined = undefined;

  @Input()
  allowEmptySearch: boolean = false;

  @Input()
  iconSearchName: IconNameEnum = IconNameEnum.send;

  @Input()
  set value(value: string) {
    this._initForm();
    this._value = value;
    this.formControl.setValue(this.value);
  }

  get value(): string {
    return isNullOrUndefined(this._value) ? StringUtil.stringEmpty : this._value;
  }

  get formControl(): FormControl {
    return this.searchForm.get(this.formDefinition.search) as FormControl;
  }

  @Input()
  mode: "auto" | "validation" = "auto";

  @Input()
  labelToTranslate: string = LabelTranslateEnum.search;

  private readonly _valueBS: BehaviorSubject<string | undefined> = new BehaviorSubject<string | undefined>(undefined);
  @Output("valueChange")
  readonly valueObs: Observable<string | undefined> = ObservableUtil.asObservable(this._valueBS);

  private readonly _blurBS: BehaviorSubject<HTMLInputElement | undefined> = new BehaviorSubject<HTMLInputElement | undefined>(undefined);
  @Output("blurChange")
  readonly blurObs: Observable<HTMLInputElement | undefined> = ObservableUtil.asObservable(this._blurBS);

  private _debounceTime: number = 600;

  constructor(protected readonly _fb: FormBuilder) {
    super();
  }

  ngOnInit(): void {
    super.ngOnInit();
    this._initForm();
    this.formControl.setValue(this.value);

    if (this.mode === this.MODE_AUTO) {
      this.subscribe(
        this.formControl.valueChanges
          .pipe(
            debounceTime(this._debounceTime),
            distinctUntilChanged(),
            tap((value: string) => {
              this._submitValue(value);
            }),
          ),
      );
    }
  }

  private _initForm(): void {
    if (!isNullOrUndefined(this.searchForm)) {
      return;
    }
    this.searchForm = this._fb.group({
      [this.formDefinition.search]: StringUtil.stringEmpty,
    });
  }

  private _submitValue(value: string): void {
    this._valueBS.next(value.trim());
  }

  isValuePresent(): boolean {
    return isNonEmptyString(this.formControl.value);
  }

  submit(): void {
    const value = this.formControl.value;
    this._submitValue(value);
  }

  clean(): void {
    this.formControl.setValue(StringUtil.stringEmpty);
  }

  blur(inputElement: HTMLInputElement): void {
    this._blurBS.next(inputElement);
  }
}

class FormComponentFormDefinition extends BaseFormDefinition {
  @PropertyName() search: string;
}
