import {
  ChangeDetectionStrategy,
  ChangeDetectorRef,
  Component,
  Input,
  OnInit,
} from "@angular/core";
import {
  AbstractControl,
  FormBuilder,
  FormControl,
  FormGroup,
  NG_VALUE_ACCESSOR,
  Validators,
} from "@angular/forms";
import {SharedAbstractPresentational} from "@app/shared/components/presentationals/shared-abstract/shared-abstract.presentational";
import {Enums} from "@enums";
import {BaseFormDefinition} from "@shared/models/base-form-definition.model";
import {
  distinctUntilChanged,
  tap,
} from "rxjs/operators";
import {
  FormValidationHelper,
  isNotNullNorUndefined,
  isNullOrUndefined,
  KeyValue,
  PropertyName,
  SolidifyValidator,
} from "solidify-frontend";

@Component({
  selector: "dlcm-shared-preservation-duration",
  templateUrl: "./shared-preservation-duration.presentational.html",
  styleUrls: ["./shared-preservation-duration.presentational.scss"],
  changeDetection: ChangeDetectionStrategy.OnPush,
  providers: [
    {
      provide: NG_VALUE_ACCESSOR,
      multi: true,
      useExisting: SharedPreservationDurationPresentational,
    },
  ],
})
export class SharedPreservationDurationPresentational extends SharedAbstractPresentational implements OnInit {
  formDefinition: FormComponentFormDefinition = new FormComponentFormDefinition();
  listUnits: KeyValue[] = Enums.RetentionPolicy.RetentionPolicyEnumTranslate;

  private readonly DAYS_BY_YEAR: number = 365;
  private readonly YEARS_BY_CENTURY: number = 100;

  @Input()
  formControl: FormControl;

  form: FormGroup;
  propagateChange = (__: any) => {};

  registerOnChange(fn: any): void {
    this.propagateChange = fn;
  }

  registerOnTouched(fn: any): void {
  }

  setDisabledState(isDisabled: boolean): void {
  }

  writeValue(value: string[]): void {
  }

  constructor(private readonly _changeDetectorRef: ChangeDetectorRef,
              private readonly _fb: FormBuilder) {
    super();
  }

  ngOnInit(): void {
    super.ngOnInit();

    const value = this.formControl.value;
    if (isNullOrUndefined(value)) {
      this.form = this._fb.group({
        [this.formDefinition.retention]: [1, [Validators.required, SolidifyValidator]],
        [this.formDefinition.retentionUnits]: [Enums.RetentionPolicy.RetentionPolicyEnum.days, [Validators.required, SolidifyValidator]],
      });
    } else {
      this.form = this._fb.group({
        [this.formDefinition.retention]: [this.adaptRetentionWithUnits(value), [Validators.required, SolidifyValidator]],
        [this.formDefinition.retentionUnits]: [this.getUnitsFromRetention(value), [Validators.required, SolidifyValidator]],
      });
    }
    this.addValidators();

    this.subscribe(this.form.get(this.formDefinition.retention).valueChanges, v => {
      this.formControl.setValue(this.calculateRetention());
      this.formControl.markAsDirty();
    });

    this.subscribe(this.form.get(this.formDefinition.retention).statusChanges, status => {
      const v = this.form.get(this.formDefinition.retention).value;
      if (status === "VALID" && isNotNullNorUndefined(v)) {
        this.formControl.setErrors(null);
      } else {
        this.formControl.setErrors({"invalid": true});
      }
    });
  }

  get retentionPolicyEnum(): typeof Enums.RetentionPolicy.RetentionPolicyEnum {
    return Enums.RetentionPolicy.RetentionPolicyEnum;
  }

  get formValidationHelper(): typeof FormValidationHelper {
    return FormValidationHelper;
  }

  getFormControl(key: string): AbstractControl {
    return FormValidationHelper.getFormControl(this.form, key);
  }

  isRequired(key: string): boolean {
    const errors = this.getFormControl(key).errors;
    return isNullOrUndefined(errors) ? false : errors.required;
  }

  calculateRetention(): number {
    const retention: number = this.form.get(this.formDefinition.retention).value;
    const units: string = this.form.get(this.formDefinition.retentionUnits).value;
    switch (units) {
      case Enums.RetentionPolicy.RetentionPolicyEnum.forever:
        return 0;
      case Enums.RetentionPolicy.RetentionPolicyEnum.days:
        return retention;
      case Enums.RetentionPolicy.RetentionPolicyEnum.years:
        return retention * this.DAYS_BY_YEAR;
      case Enums.RetentionPolicy.RetentionPolicyEnum.centuries:
        return retention * this.DAYS_BY_YEAR * this.YEARS_BY_CENTURY;
      default:
        return;
    }
  }

  private getUnitsFromRetention(retention: number): string {
    let unit: string;
    if (retention === 0) {
      unit = Enums.RetentionPolicy.RetentionPolicyEnum.forever;
    } else if (retention < this.DAYS_BY_YEAR) {
      unit = Enums.RetentionPolicy.RetentionPolicyEnum.days;
    } else if (retention < (this.DAYS_BY_YEAR * this.YEARS_BY_CENTURY)) {
      unit = Enums.RetentionPolicy.RetentionPolicyEnum.years;
    } else {
      unit = Enums.RetentionPolicy.RetentionPolicyEnum.centuries;
    }
    return unit;
  }

  private adaptRetentionWithUnits(retention: number): number {
    let newRetention: number;
    if (retention === 0) {
      newRetention = 0;
    } else if (retention < this.DAYS_BY_YEAR) {
      return retention;
    } else if (retention >= this.DAYS_BY_YEAR && retention < (this.DAYS_BY_YEAR * this.YEARS_BY_CENTURY)) {
      newRetention = retention / this.DAYS_BY_YEAR;
    } else {
      newRetention = retention / (this.DAYS_BY_YEAR * this.YEARS_BY_CENTURY);
    }
    return newRetention;
  }

  private addValidators(): void {
    const retentionUnitsFormControl = this.form.get(this.formDefinition.retentionUnits);
    this.subscribe(retentionUnitsFormControl.valueChanges.pipe(
      distinctUntilChanged(),
      tap(value => {
        this.setValidator(value);
      }),
    ));
    this.setValidator(retentionUnitsFormControl.value);
  }

  private setValidator(value: Enums.RetentionPolicy.RetentionPolicyEnum): void {
    const retentionFormControl = this.form.get(this.formDefinition.retention);
    if (value === Enums.RetentionPolicy.RetentionPolicyEnum.days) {
      retentionFormControl.setValidators([Validators.min(1), Validators.max(this.DAYS_BY_YEAR)]);
    } else if (value === Enums.RetentionPolicy.RetentionPolicyEnum.years) {
      retentionFormControl.setValidators([Validators.min(1), Validators.max(this.YEARS_BY_CENTURY)]);
    } else if (value === Enums.RetentionPolicy.RetentionPolicyEnum.centuries) {
      retentionFormControl.setValidators([Validators.min(1)]);
    } else {
      retentionFormControl.setValue(0);
      retentionFormControl.setValidators([]);
    }
    retentionFormControl.updateValueAndValidity();
    this._changeDetectorRef.detectChanges();
  }
}

class FormComponentFormDefinition extends BaseFormDefinition {
  @PropertyName() retention: string;
  @PropertyName() retentionUnits: string;
}
