import {
  ChangeDetectionStrategy,
  ChangeDetectorRef,
  Component,
  ElementRef,
  Input,
} from "@angular/core";
import {
  FormBuilder,
  Validators,
} from "@angular/forms";
import {Enums} from "@enums";
import {
  NotificationDlcm,
  Role,
} from "@models";
import {SharedAbstractFormPresentational} from "@shared/components/presentationals/shared-abstract-form/shared-abstract-form.presentational";
import {BaseResourceApiEnum} from "@shared/enums/api.enum";
import {
  AppRoutesEnum,
  DepositRoutesEnum,
  PreservationPlanningRoutesEnum,
  RoutesEnum,
  SharedAipRoutesEnum,
} from "@shared/enums/routes.enum";
import {BaseFormDefinition} from "@shared/models/base-form-definition.model";
import {Storage} from "@shared/models/storage.model";
import {BreakpointService} from "@shared/services/breakpoint.service";
import {SecurityService} from "@shared/services/security.service";
import {sharedDepositActionNameSpace} from "@shared/stores/deposit/shared-deposit.action";
import {SharedDepositState} from "@shared/stores/deposit/shared-deposit.state";
import {sharedOrganizationalUnitActionNameSpace} from "@shared/stores/organizational-unit/shared-organizational-unit.action";
import {SharedOrganizationalUnitState} from "@shared/stores/organizational-unit/shared-organizational-unit.state";
import {sharedPersonActionNameSpace} from "@shared/stores/person/shared-person.action";
import {SharedPersonState} from "@shared/stores/person/shared-person.state";
import {
  DateUtil,
  KeyValue,
  PropertyName,
  ResourceNameSpace,
  SolidifyValidator,
  urlSeparator,
} from "solidify-frontend";

@Component({
  selector: "dlcm-shared-notification-form",
  templateUrl: "./shared-notification-form.presentational.html",
  styleUrls: ["../shared-abstract-form/shared-abstract-form.presentational.scss"],
  changeDetection: ChangeDetectionStrategy.OnPush,
})
export class SharedNotificationFormPresentational extends SharedAbstractFormPresentational<NotificationDlcm> {
  formDefinition: FormComponentFormDefinition = new FormComponentFormDefinition();

  listNotificationCategory: KeyValue[] = Enums.Notification.CategoryEnumTranslate;
  listNotificationType: KeyValue[] = Enums.Notification.TypeEnumTranslate;
  listNotificationStatusTranslate: KeyValue[] = Enums.Notification.StatusEnumTranslate;

  get notificationTypeEnum(): typeof Enums.Notification.TypeEnum {
    return Enums.Notification.TypeEnum;
  }

  get defaultAipStorage(): Storage {
    return BaseResourceApiEnum.archivalStorageDefault;
  }

  @Input()
  listRole: Role[];

  sharedPersonActionNameSpace: ResourceNameSpace = sharedPersonActionNameSpace;
  sharedPersonState: typeof SharedPersonState = SharedPersonState;

  sharedOrgUnitActionNameSpace: ResourceNameSpace = sharedOrganizationalUnitActionNameSpace;
  sharedOrganizationalUnitState: typeof SharedOrganizationalUnitState = SharedOrganizationalUnitState;

  sharedDepositActionNameSpace: ResourceNameSpace = sharedDepositActionNameSpace;
  sharedDepositState: typeof SharedDepositState = SharedDepositState;

  constructor(protected readonly _changeDetectorRef: ChangeDetectorRef,
              protected readonly _elementRef: ElementRef,
              private readonly _fb: FormBuilder,
              public readonly securityService: SecurityService,
              public readonly breakpointService: BreakpointService) {
    super(_changeDetectorRef, _elementRef);
  }

  protected bindFormTo(notificationDlcm: NotificationDlcm): void {
    this.form = this._fb.group({
      [this.formDefinition.notificationCategory]: [notificationDlcm.notificationType?.notificationCategory, [Validators.required, SolidifyValidator]],
      [this.formDefinition.notificationType]: [notificationDlcm.notificationType?.resId, [Validators.required, SolidifyValidator]],
      [this.formDefinition.notificationStatus]: [notificationDlcm.notificationStatus, [Validators.required, SolidifyValidator]],
      [this.formDefinition.emitterId]: [notificationDlcm.emitter?.person?.resId, [Validators.required, SolidifyValidator]],
      [this.formDefinition.message]: [notificationDlcm.message, [Validators.required, SolidifyValidator]],
      [this.formDefinition.objectId]: [notificationDlcm.objectId, [Validators.required, SolidifyValidator]],
      [this.formDefinition.creation]: [DateUtil.convertDateToDateTimeString(new Date(notificationDlcm.creation.when)), [Validators.required, SolidifyValidator]],
      [this.formDefinition.lastUpdate]: [DateUtil.convertDateToDateTimeString(new Date(notificationDlcm.lastUpdate.when)), [Validators.required, SolidifyValidator]],
      [this.formDefinition.notifiedOrgUnitId]: [notificationDlcm.notifiedOrgUnit?.resId, [Validators.required, SolidifyValidator]],
    });
  }

  protected initNewForm(): void {
    this.form = this._fb.group({
      [this.formDefinition.notificationCategory]: ["", [Validators.required, SolidifyValidator]],
      [this.formDefinition.notificationType]: ["", [Validators.required, SolidifyValidator]],
      [this.formDefinition.notificationStatus]: ["", [Validators.required, SolidifyValidator]],
      [this.formDefinition.emitterId]: ["", [Validators.required, SolidifyValidator]],
      [this.formDefinition.message]: ["", [Validators.required, SolidifyValidator]],
      [this.formDefinition.objectId]: ["", [Validators.required, SolidifyValidator]],
      [this.formDefinition.creation]: ["", [Validators.required, SolidifyValidator]],
      [this.formDefinition.lastUpdate]: ["", [Validators.required, SolidifyValidator]],
      [this.formDefinition.notifiedOrgUnitId]: ["", [Validators.required, SolidifyValidator]],
    });
  }

  protected treatmentBeforeSubmit(notificationDlcm: NotificationDlcm): NotificationDlcm {
    return notificationDlcm;
  }

  navigateToOrgUnit(orgUnitId: string): void {
    this.navigate([RoutesEnum.preservationSpaceOrganizationalUnitDetail, orgUnitId]);
  }

  navigateToDeposit(depositId: string): void {
    this.navigate([RoutesEnum.deposit + urlSeparator + this.model.notifiedOrgUnit.resId + urlSeparator + DepositRoutesEnum.detail + urlSeparator + depositId]);
  }

  navigateToPerson(personId: string): void {
    this.navigate([RoutesEnum.adminPersonDetail, personId]);
  }

  goToAip(): void {
    const aipId: string = this.model.objectId;
    this._navigateBS.next([AppRoutesEnum.preservationPlanning, PreservationPlanningRoutesEnum.aip, String(this.defaultAipStorage.index), SharedAipRoutesEnum.aipDetail, aipId]);
  }

  goToAipDownloaded(): void {
    const aipId: string = this.model.objectId;
    this._navigateBS.next([RoutesEnum.preservationPlanningAipDownloadedDetail, aipId]);
  }

  goToSip(): void {
    const sipId: string = this.model.objectId;
    this._navigateBS.next([RoutesEnum.preservationPlanningSipDetail, sipId]);
  }

  goToDip(): void {
    const dipId: string = this.model.objectId;
    this._navigateBS.next([RoutesEnum.preservationPlanningDipDetail, dipId]);
  }

  goToAipSteward(): void {
    const aipId: string = this.model.objectId;
    this._navigateBS.next([RoutesEnum.preservationSpaceAipStewardDetail, aipId]);
  }

  goToArchive(): void {
    const archiveId: string = this.model.objectId;
    this._navigateBS.next([RoutesEnum.homeDetail, archiveId]);
  }
}

class FormComponentFormDefinition extends BaseFormDefinition {
  @PropertyName() notificationCategory: string;
  @PropertyName() notificationType: string;
  @PropertyName() notificationStatus: string;
  @PropertyName() emitterId: string;
  @PropertyName() message: string;
  @PropertyName() objectId: string;
  @PropertyName() creation: string;
  @PropertyName() lastUpdate: string;
  @PropertyName() notifiedOrgUnitId: string;
}
