import {Overlay} from "@angular/cdk/overlay";
import {ComponentPortal} from "@angular/cdk/portal";
import {
  ChangeDetectionStrategy,
  ChangeDetectorRef,
  Component,
  ElementRef,
  Input,
  OnInit,
  ViewContainerRef,
} from "@angular/core";
import {
  ControlValueAccessor,
  FormBuilder,
  FormControl,
  NG_VALUE_ACCESSOR,
} from "@angular/forms";
import {SharedAbstractMultiSelectPresentational} from "@shared/components/presentationals/shared-abstract-multi-select/shared-abstract-multi-select.presentational";
import {SharedMultiSelectContentPresentational} from "@shared/components/presentationals/shared-multi-select-content/shared-multi-select-content.presentational";
import {SharedMultiSelectDefaultValueContentPresentational} from "@shared/components/presentationals/shared-multi-select-default-value-content/shared-multi-select-default-value-content.presentational";
import {TabulationService} from "@shared/services/tabulation.service";
import {distinctUntilChanged} from "rxjs/operators";
import {
  BaseResourceType,
  isEmptyString,
  isNonEmptyArray,
  isNullOrUndefined,
  NotificationService,
  Override,
} from "solidify-frontend";

@Component({
  selector: "dlcm-shared-multi-select",
  templateUrl: "./shared-multi-select.presentational.html",
  styleUrls: ["./shared-multi-select.presentational.scss"],
  changeDetection: ChangeDetectionStrategy.OnPush,
  providers: [
    {
      provide: NG_VALUE_ACCESSOR,
      multi: true,
      useExisting: SharedMultiSelectPresentational,
    },
  ],
})
export class SharedMultiSelectPresentational<TResource extends BaseResourceType> extends SharedAbstractMultiSelectPresentational<TResource> implements ControlValueAccessor, OnInit {
  @Input()
  list: TResource[];

  @Input()
  formControlDefault: FormControl | undefined;

  get isWithDefaultValue(): boolean {
    return !isNullOrUndefined(this.formControlDefault);
  }

  constructor(protected _fb: FormBuilder,
              protected _elementRef: ElementRef,
              protected _viewContainerRef: ViewContainerRef,
              protected _overlay: Overlay,
              protected _changeDetector: ChangeDetectorRef,
              protected _notificationService: NotificationService,
              protected _tabulationService: TabulationService) {
    super(_fb,
      _elementRef,
      _viewContainerRef,
      _overlay,
      _changeDetector,
      _notificationService,
      _tabulationService);
  }

  @Override()
  ngOnInit(): void {
    super.ngOnInit();
    if (this.isWithDefaultValue) {
      this.subscribe(this.formControlDefault.valueChanges.pipe(distinctUntilChanged()), () => this._changeDetector.detectChanges());
    }
  }

  @Override()
  remove(item: TResource): void {
    super.remove(item);
    if (this.isWithDefaultValue) {
      const defaultValueId = this.formControlDefault.value;
      if (item.resId === defaultValueId) {
        const listValueSelected = this.formControl.value;
        if (listValueSelected.length > 0) {
          this.formControlDefault.setValue(listValueSelected[0]);
        } else {
          this.formControlDefault.setValue(undefined);
        }
        this.formControlDefault.markAsTouched();
      }
    }
  }

  add(item: TResource): void {
    super.add(item);
    if (this.isWithDefaultValue) {
      const defaultValueId = this.formControlDefault.value;
      if (isNullOrUndefined(defaultValueId) || isEmptyString(defaultValueId)) {
        const listValueSelected = this.formControl.value;
        if (listValueSelected.length > 0) {
          this.formControlDefault.setValue(listValueSelected[0]);
        }
        this.formControlDefault.markAsTouched();
      }
    }
  }

  protected _manageInitExistingValue(): void {
    if (isNullOrUndefined(this.list)) {
      return;
    }
    const keys = this.formControl.value;
    if (isNonEmptyArray(keys)) {
      this.selectedItems = [];
      keys.forEach(key => {
        const element = this.list.find(item => item.resId === key);
        if (isNullOrUndefined(element)) {
          throw new Error(`Element with key ${key} not find in list for multi-select '${this.title}'`);
        }
        this.selectedItems.push(element);
      });
    }
  }

  getComponentPortal(): ComponentPortal<SharedMultiSelectDefaultValueContentPresentational<TResource>> | ComponentPortal<SharedMultiSelectContentPresentational<TResource>> {
    if (this.isWithDefaultValue) {
      return new ComponentPortal<SharedMultiSelectDefaultValueContentPresentational<TResource>>(SharedMultiSelectDefaultValueContentPresentational);
    }
    return new ComponentPortal<SharedMultiSelectContentPresentational<TResource>>(SharedMultiSelectContentPresentational);
  }

  protected extraBehaviorOnValueSelectedInOvertly(value: TResource): void {
  }
}
