import {
  BACKSLASH,
  COMMA,
  ENTER,
  SEMICOLON,
  SLASH,
  SPACE,
} from "@angular/cdk/keycodes";
import {
  ChangeDetectionStrategy,
  ChangeDetectorRef,
  Component,
  Input,
  OnInit,
  Output,
} from "@angular/core";
import {
  ControlValueAccessor,
  FormControl,
  NG_VALUE_ACCESSOR,
} from "@angular/forms";
import {MatChipInputEvent} from "@angular/material/chips";
import {MatFormFieldAppearance} from "@angular/material/form-field";
import {SharedAbstractPresentational} from "@shared/components/presentationals/shared-abstract/shared-abstract.presentational";
import {LabelTranslateEnum} from "@shared/enums/label-translate.enum";
import {
  BehaviorSubject,
  Observable,
} from "rxjs";
import {
  distinctUntilChanged,
  tap,
} from "rxjs/operators";
import {
  isArray,
  isEmptyString,
  isNullOrUndefined,
  ObservableUtil,
  StringUtil,
} from "solidify-frontend";

@Component({
  selector: "dlcm-shared-keyword-input",
  templateUrl: "./shared-keyword-input.presentational.html",
  styleUrls: ["./shared-keyword-input.presentational.scss"],
  changeDetection: ChangeDetectionStrategy.OnPush,
  providers: [
    {
      provide: NG_VALUE_ACCESSOR,
      multi: true,
      useExisting: SharedKeywordInputPresentational,
    },
  ],
})

export class SharedKeywordInputPresentational extends SharedAbstractPresentational implements OnInit, ControlValueAccessor {
  readonly separatorKeysCodes: number[] = [ENTER, SPACE, COMMA, SEMICOLON, SLASH, BACKSLASH];

  @Input()
  placeholderToTranslate: string = LabelTranslateEnum.keywords;

  @Input()
  readonly: boolean;

  @Input()
  formControl: FormControl;

  @Input()
  appearance: MatFormFieldAppearance = this.appearanceInputMaterial;

  private readonly _valueBS: BehaviorSubject<string[] | undefined> = new BehaviorSubject<string[] | undefined>(undefined);
  @Output("valueChange")
  readonly valueObs: Observable<string[] | undefined> = ObservableUtil.asObservable(this._valueBS);

  constructor(private readonly _changeDetector: ChangeDetectorRef) {
    super();
  }

  ngOnInit(): void {
    this.subscribe(this.formControl.valueChanges.pipe(
      distinctUntilChanged(),
      tap(value => {
        this.propagateChange(value);
        this._valueBS.next(value);
        this.formControl.markAsPristine();
        this._changeDetector.detectChanges();
      }),
    ));
  }

  propagateChange = (__: any) => {};

  registerOnChange(fn: any): void {
    this.propagateChange = fn;
  }

  registerOnTouched(fn: any): void {
  }

  setDisabledState(isDisabled: boolean): void {
  }

  writeValue(value: string[]): void {
  }

  remove(value: string): void {
    const list: string[] = this.formControl.value;
    const indexOf = list.indexOf(value);
    if (indexOf === -1) {
      return;
    }
    list.splice(indexOf, 1);
    this.formControl.markAsDirty();
  }

  add(event: MatChipInputEvent): void {
    const input = event.input;
    const value = (event.value || StringUtil.stringEmpty).trim();
    const list: string[] = this.formControl.value;

    if (!isEmptyString(value) && !list.includes(value)) {
      list.push(value.trim());
      this.formControl.markAsDirty();
    }

    if (input) {
      input.value = StringUtil.stringEmpty;
    }
  }

  getValue(): string[] {
    const value = this.formControl.value;
    return (!isNullOrUndefined(value) && isArray(value)) ? value.filter(v => v !== StringUtil.stringEmpty) : [];
  }
}
