import {CdkDragDrop} from "@angular/cdk/drag-drop";
import {FlatTreeControl} from "@angular/cdk/tree";
import {
  AfterViewInit,
  ChangeDetectionStrategy,
  Component,
  Input,
  OnInit,
  Output,
} from "@angular/core";
import {
  MatTreeFlatDataSource,
  MatTreeFlattener,
} from "@angular/material/tree";
import {DepositDataFileHelper} from "@deposit/helpers/deposit-data-file.helper";
import {DepositDataFile} from "@deposit/models/deposit-data-file.model";
import {environment} from "@environments/environment";
import {SharedAbstractPresentational} from "@shared/components/presentationals/shared-abstract/shared-abstract.presentational";
import {
  BehaviorSubject,
  Observable,
} from "rxjs";
import {
  isNotNullNorUndefined,
  isNullOrUndefined,
  ObjectUtil,
  ObservableUtil,
  StringUtil,
} from "solidify-frontend";

export class FlatNode {
  constructor(public folderName: string,
              public fullFolderName: string,
              public level: number = 0,
              public expandable: boolean = false,
              public postCreation: boolean = false) {}

}

@Component({
  selector: "dlcm-shared-folder-tree",
  templateUrl: "./shared-folder-tree.presentational.html",
  styleUrls: ["./shared-folder-tree.presentational.scss"],
  changeDetection: ChangeDetectionStrategy.OnPush,
})
export class SharedFolderTreePresentational extends SharedAbstractPresentational implements OnInit, AfterViewInit {
  private readonly _SEPARATOR: string = DepositDataFileHelper.SEPARATOR;
  private readonly _ROOT: string = DepositDataFileHelper.ROOT;
  readonly indentation: number = 10;
  readonly CDK_DROP_LIST_PREFIX: string = environment.cdkDropListIdPrefix;
  readonly TOOLTIP_NODE_NAME_DELAY: number = 500;

  expandedNodes: string[] = [];

  @Input()
  currentFolder: string | undefined;

  private _intermediateFolders: string[];

  @Input()
  set intermediateFolders(value: string[]) {
    this._intermediateFolders = value;
    this.triggerInitIfReady();
  }

  get intermediateFolders(): string[] {
    return this._intermediateFolders;
  }

  private readonly _selectBS: BehaviorSubject<string> = new BehaviorSubject<string>(undefined);
  @Output("selectChange")
  readonly selectObs: Observable<string> = ObservableUtil.asObservable(this._selectBS);

  private readonly _moveBS: BehaviorSubject<DepositDataFile> = new BehaviorSubject<DepositDataFile>(undefined);
  @Output("moveChange")
  readonly moveObs: Observable<DepositDataFile> = ObservableUtil.asObservable(this._moveBS);

  private readonly _downloadFolderBS: BehaviorSubject<string> = new BehaviorSubject<string>(undefined);
  @Output("downloadFolder")
  readonly downloadFolderObs: Observable<string> = ObservableUtil.asObservable(this._downloadFolderBS);

  private readonly _deleteFolderBS: BehaviorSubject<string> = new BehaviorSubject<string>(undefined);
  @Output("deleteFolder")
  readonly deleteFolderObs: Observable<string> = ObservableUtil.asObservable(this._deleteFolderBS);

  private readonly _uploadInFolderBS: BehaviorSubject<string> = new BehaviorSubject<string>(undefined);
  @Output("uploadInFolder")
  readonly uploadInFolderObs: Observable<string> = ObservableUtil.asObservable(this._uploadInFolderBS);

  @Input()
  isLoading: boolean;

  @Input()
  expandFirstLevel: boolean;

  @Input()
  canDelete: boolean = false;

  @Input()
  canDownload: boolean = true;

  @Input()
  draggingDepositDataFile: DepositDataFile | undefined;

  isReady: boolean = false;

  private _foldersWithIntermediateFolders: string[];

  @Input()
  set foldersWithIntermediateFolders(value: string[]) {
    this._foldersWithIntermediateFolders = value;
    if (this.isReady) {
      this.setupFoldersWithIntermediateFolders();
    } else {
      this.triggerInitIfReady();
    }
  }

  private init(): void {
    this.setupFoldersWithIntermediateFolders();
  }

  private setupFoldersWithIntermediateFolders(): void {
    this.saveExpandedNodes();
    if (isNullOrUndefined(this._foldersWithIntermediateFolders)) {
      return;
    }
    const listFolders = [...this._foldersWithIntermediateFolders];
    this.dataSource.data = listFolders;
    this.expandFirstLevelTree();
    this.restoreExpandedNodes();
  }

  saveExpandedNodes(): void {
    this.expandedNodes = [];
    if (isNullOrUndefined(this.treeControl.dataNodes)) {
      return;
    }
    this.treeControl.dataNodes.forEach(node => {
      if (node.expandable && this.treeControl.isExpanded(node)) {
        this.expandedNodes.push(node.fullFolderName);
      }
    });
  }

  restoreExpandedNodes(): void {
    this.expandedNodes.forEach(node => {
      this.treeControl.expand(this.treeControl.dataNodes.find(n => n.fullFolderName === node));
    });
  }

  private _transformer = (fullFolderName: string): FlatNode => ({
    expandable: this.existChild(fullFolderName),
    folderName: this.getFolderName(fullFolderName),
    fullFolderName: fullFolderName,
    level: this.getLevel(fullFolderName),
    postCreation: this.isIntermediateFolderCreate(fullFolderName),
  }); // tslint:disable-line

  treeFlattener: MatTreeFlattener<string, FlatNode>;
  treeControl: FlatTreeControl<FlatNode>;
  dataSource: MatTreeFlatDataSource<string, FlatNode>;

  expandedAll: boolean = true;
  eventMouseOver: MouseEvent | undefined = undefined;

  constructor() {
    super();
  }

  ngOnInit(): void {
    if (this.isReady) {
      return;
    }
    this.treeControl = new FlatTreeControl<FlatNode>(node => node.level, node => node.expandable);
    this.treeFlattener = new MatTreeFlattener(this._transformer, node => node.level, node => node.expandable, node => []/*node.children*/);
    this.dataSource = new MatTreeFlatDataSource(this.treeControl, this.treeFlattener);
    this.triggerInitIfReady();
  }

  triggerInitIfReady(): void {
    if (this.isReady === false) {
      if (isNotNullNorUndefined(this.treeControl) && isNotNullNorUndefined(this.treeFlattener) && isNotNullNorUndefined(this.dataSource) && isNotNullNorUndefined(this.intermediateFolders) && isNotNullNorUndefined(this.foldersWithIntermediateFolders)) {
        this.isReady = true;
        this.init();
      }
    }
  }

  ngAfterViewInit(): void {
    super.ngAfterViewInit();
    if (isNullOrUndefined(this.treeControl.dataNodes) || this.treeControl.dataNodes.length === 0) {
      return;
    }
    this.expandFirstLevelTree();
  }

  private expandFirstLevelTree(): void {
    if (this.currentFolder !== this._ROOT) {
      const node = this.treeControl.dataNodes.find(d => d.fullFolderName === this.currentFolder);
      this.expandRecursivelyParent(node);
      this.treeControl.expand(node);
      return;
    }
    if (this.expandFirstLevelTree) {
      const firstNode = this.treeControl.dataNodes[0];
      this.treeControl.expand(firstNode);
    }
  }

  private expandRecursivelyParent(node: FlatNode): void {
    if (isNullOrUndefined(node)) {
      return;
    }
    if (node.fullFolderName === this._ROOT) {
      return;
    }
    let parentPath = node.fullFolderName.substring(0, node.fullFolderName.lastIndexOf(this._SEPARATOR));
    if (parentPath === StringUtil.stringEmpty) {
      parentPath = this._ROOT;
    }
    const parentNode = this.treeControl.dataNodes.find(d => d.fullFolderName === parentPath);
    this.treeControl.expand(parentNode);
    this.expandRecursivelyParent(parentNode);
  }

  expandAllOrCollapseAll(): void {
    const parentNode = this.treeControl.dataNodes.find(d => d.folderName === this._ROOT);
    if (this.treeControl.isExpanded(parentNode)) {
      this.treeControl.collapseAll();
      this.expandedAll = false;
    } else {
      this.treeControl.expandAll();
      this.expandedAll = true;
    }
  }

  get foldersWithIntermediateFolders(): string[] {
    return this._foldersWithIntermediateFolders;
  }

  getFolderName(fullFolderName: string): string {
    if (fullFolderName === this._ROOT) {
      return fullFolderName;
    }
    const splitted = fullFolderName.split(this._SEPARATOR);
    return splitted[splitted.length - 1];
  }

  getLevel(fullFolderName: string): number {
    if (fullFolderName === this._SEPARATOR) {
      return 0;
    }
    const splitted = fullFolderName.split(this._SEPARATOR);
    return splitted.length - 1;
  }

  hasChild = (_: number, node: FlatNode): boolean => node.expandable;

  select(node: FlatNode): void {
    if (node.fullFolderName === this.currentFolder) {
      return;
    }
    this.currentFolder = node.fullFolderName;
    this._selectBS.next(node.fullFolderName);
  }

  private existChild(currentFullFolderName: string): boolean {
    if (currentFullFolderName === this._ROOT) {
      return true;
    }
    const item = this.dataSource.data.find(fullFolderName => fullFolderName.startsWith(currentFullFolderName + this._SEPARATOR));
    return !isNullOrUndefined(item);
  }

  private isIntermediateFolderCreate(fullFolderName: string): boolean {
    return isNullOrUndefined(this.intermediateFolders) || this.intermediateFolders.includes(fullFolderName);
  }

  drop($event: CdkDragDrop<any, any>, folderFullName: string): void {
    const depositDataFile = ObjectUtil.clone($event.item.data as DepositDataFile);
    if (depositDataFile.relativeLocation === folderFullName) {
      return;
    }
    depositDataFile.relativeLocation = folderFullName;
    this._moveBS.next(depositDataFile);
  }

  openFolder(node: FlatNode): void {
    // console.error("ENTER", node);
    // this.treeControl.expand(node);

    // TODO Need to add to dropzone list child node of current node expanded
    // this.treeControl.expand(node);
  }

  downloadFolder(node: FlatNode): void {
    this._downloadFolderBS.next(node.fullFolderName);
  }

  deleteFolder(node: FlatNode): void {
    this._deleteFolderBS.next(node.fullFolderName);
  }

  upload(node: FlatNode): void {
    this._uploadInFolderBS.next(node.fullFolderName);
  }

  mouseOver($event: MouseEvent, node: FlatNode): void {
    const dragInProgress = isNotNullNorUndefined(this.draggingDepositDataFile);
    if (dragInProgress) {
      this.eventMouseOver = $event;
      setTimeout(() => {
        if ($event === this.eventMouseOver) {
          this.treeControl.expand(node);
        }
      }, 500);
    }
  }

  mouseOut($event: MouseEvent, node: FlatNode): void {
    this.eventMouseOver = undefined;
  }
}
