import {
  ChangeDetectionStrategy,
  Component,
  HostBinding,
  HostListener,
  Input,
} from "@angular/core";
import {MatDialog} from "@angular/material/dialog";
import {Enums} from "@enums";
import {
  SharedInfoExcludedIgnoredFileDialog,
  SharedInfoExcludedIgnoredFileDialogData,
} from "@shared/components/dialogs/shared-info-excluded-ignored-file/shared-info-excluded-ignored-file.dialog";
import {SharedAbstractPresentational} from "@shared/components/presentationals/shared-abstract/shared-abstract.presentational";

@Component({
  selector: "dlcm-shared-datafile-quick-status",
  templateUrl: "./shared-datafile-quick-status.presentational.html",
  styleUrls: ["./shared-datafile-quick-status.presentational.scss"],
  changeDetection: ChangeDetectionStrategy.OnPush,
})

export class SharedDatafileQuickStatusPresentational extends SharedAbstractPresentational {
  private readonly CLASS_READY: string = "ready";
  private readonly CLASS_ERROR: string = "error";
  private readonly CLASS_WARNING: string = "warning";
  private readonly CLASS_CLEANED: string = "cleaned";
  private readonly CLASS_CAN_SEE_DETAIL: string = "can-see-detail";

  readonly CLASS_PENDING: string = "pending";

  _value: Enums.DataFile.StatusEnum;

  @Input()
  set value(value: Enums.DataFile.StatusEnum) {
    this._value = value;
    this.classes = this.getColor();
    this.toolipToTranslate = Enums.DataFile.getExplanation(value);
  }

  get value(): Enums.DataFile.StatusEnum {
    return this._value;
  }

  constructor(private readonly _dialog: MatDialog) {
    super();
  }

  toolipToTranslate: string | undefined;

  @HostBinding("class")
  classes: string;

  @HostListener("click", ["$event"]) click(mouseEvent: MouseEvent): void {
    mouseEvent.stopPropagation();
    if (this.value !== Enums.DataFile.StatusEnum.EXCLUDED_FILE && this.value !== Enums.DataFile.StatusEnum.IGNORED_FILE) {
      return;
    }
    this._dialog.open(SharedInfoExcludedIgnoredFileDialog, {
      data: {
        dataFileStatusEnum: this.value,
      } as SharedInfoExcludedIgnoredFileDialogData,
    });
  }

  getColor(): string {
    switch (this.value) {
      case Enums.DataFile.StatusEnum.READY:
        return this.CLASS_READY;
      case Enums.DataFile.StatusEnum.IN_ERROR:
        return this.CLASS_ERROR;
      case Enums.DataFile.StatusEnum.EXCLUDED_FILE:
        return this.CLASS_ERROR + " " + this.CLASS_CAN_SEE_DETAIL;
      case Enums.DataFile.StatusEnum.IGNORED_FILE:
        return this.CLASS_WARNING + " " + this.CLASS_CAN_SEE_DETAIL;
      case Enums.DataFile.StatusEnum.CLEANED:
        return this.CLASS_CLEANED;
      default:
        return this.CLASS_PENDING;
    }
  }
}
