import {
  CdkDragEnd,
  CdkDragStart,
} from "@angular/cdk/drag-drop";
import {
  ChangeDetectionStrategy,
  ChangeDetectorRef,
  Component,
  ElementRef,
  HostBinding,
  Input,
  OnChanges,
  OnInit,
  Output,
  SimpleChanges,
  TrackByFunction,
  ViewChild,
} from "@angular/core";
import {FormControl} from "@angular/forms";
import {MatCheckboxChange} from "@angular/material/checkbox";
import {MatDialog} from "@angular/material/dialog";
import {MatSelect} from "@angular/material/select";
import {SharedAbstractPresentational} from "@app/shared/components/presentationals/shared-abstract/shared-abstract.presentational";
import {SharedPaginatorPresentational} from "@app/shared/components/presentationals/shared-paginator/shared-paginator.presentational";
import {FieldTypeEnum} from "@app/shared/enums/field-type.enum";
import {DataTableColumns} from "@app/shared/models/data-table-columns.model";
import {environment} from "@environments/environment";
import {
  SharedDataTableLooseSelectionDialog,
  SharedDataTableLooseSelectionDialogData,
} from "@shared/components/dialogs/shared-data-table-loose-selection/shared-data-table-loose-selection.dialog";
import {DataTableComponentEnum} from "@shared/enums/data-table-component.enum";
import {IconNameEnum} from "@shared/enums/icon-name.enum";
import {DataTableActions} from "@shared/models/data-table-actions.model";
import {DataTableBulkActions} from "@shared/models/data-table-bulk-actions.model";
import {
  ExtraButtonToolbar,
  ObservableOrPromiseOrValue,
} from "@shared/models/extra-button-toolbar.model";
import {BreakpointService} from "@shared/services/breakpoint.service";
import {isNotNullOrUndefined} from "codelyzer/util/isNotNullOrUndefined";
import {
  BehaviorSubject,
  Observable,
} from "rxjs";
import {
  map,
  take,
  tap,
} from "rxjs/operators";
import {
  BaseResource,
  DateUtil,
  EnumUtil,
  isEmptyArray,
  isEmptyString,
  isFalse,
  isFunction,
  isNonEmptyArray,
  isNonEmptyString,
  isNotNullNorUndefined,
  isNullOrUndefined,
  isObservable,
  isString,
  isTrue,
  MappingObject,
  MappingObjectUtil,
  MARK_AS_TRANSLATABLE,
  NotificationService,
  ObjectUtil,
  ObservableUtil,
  OrderEnum,
  Paging,
  QueryParameters,
  QueryParametersUtil,
  Sort,
  StringUtil,
} from "solidify-frontend";

@Component({
  selector: "dlcm-shared-data-table",
  templateUrl: "./shared-data-table.presentational.html",
  styleUrls: ["./shared-data-table.presentational.scss"],
  changeDetection: ChangeDetectionStrategy.OnPush,
})
export class SharedDataTablePresentational<TResource extends BaseResource> extends SharedAbstractPresentational implements OnInit, OnChanges {
  context: Context = new Context();

  readonly TIME_BEFORE_DISPLAY_TOOLTIP: number = environment.timeBeforeDisplayTooltipOnDataTable;

  private _datas: TResource[];
  currentSort: Sort;

  @HostBinding("attr.data-test-info-number-line")
  numberLine: number;

  @ViewChild("headerTitleLine")
  headerTitleLine: ElementRef;

  @ViewChild("thead")
  thead: ElementRef;

  @ViewChild("tbody")
  tbody: ElementRef;

  @Input()
  set datas(value: TResource[]) {
    if (this.pendingRetreiveAllForSelection) {
      this.pendingRetreiveAllForSelection = false;
      this.multiSelectionValueOnAllPages = value;
      this.computeNumberItemSelectedOnCurrentPage();
      this.displaySelectAllElementSuccess = true;
      return;
    }
    this._datas = value;
    this.computeIsDatasPresent();
    this.computeNumberItemSelectedOnCurrentPage();
  }

  get datas(): TResource[] {
    return this._datas;
  }

  ngOnInit(): void {
    super.ngOnInit();

    this.backendSort(null);
  }

  private _datasForPage: TResource[];

  get datasForPage(): TResource[] {
    return this._datasForPage;
  }

  forceHoverRowData: TResource | undefined;

  trackByFunction: TrackByFunction<TResource> = (index: number, item: TResource) => index;

  @HostBinding("class.is-clickable")
  @Input()
  isClickable: boolean = true;

  @ViewChild("headerButtonAndPagination")
  headerButtonAndPagination: ElementRef;

  private _HEIGHT_PAGINATOR: number = 50;
  private _HEIGHT_HEADER_TITLE: number = 40;
  private _STICKY_FIX_GLITCH_CONTENT_VISIBLE_ON_TOP_OF_HEADER: number = -1;
  private _STICKY_FIX_GLITCH_HORIZONTAL_LINE: number = -1;

  @Input()
  stickyTopPosition: number | undefined = environment.defaultStickyDatatableHeight;

  get stickyTopPositionText(): string {
    return this.stickyTopPosition + "px";
  }

  get stickyTopPositionTitleNumber(): number {
    return this.stickyTopPosition + (this.headerButtonAndPagination?.nativeElement ? this.headerButtonAndPagination.nativeElement.offsetHeight : this._HEIGHT_PAGINATOR) + this._STICKY_FIX_GLITCH_CONTENT_VISIBLE_ON_TOP_OF_HEADER;
  }

  get stickyTopPositionTitleText(): string {
    return this.stickyTopPositionTitleNumber + "px";
  }

  get stickyTopPositionFilterNumber(): number {
    return this.stickyTopPositionTitleNumber + (this.headerTitleLine?.nativeElement ? this.headerTitleLine.nativeElement.offsetHeight : this._HEIGHT_HEADER_TITLE) + this._STICKY_FIX_GLITCH_HORIZONTAL_LINE;
  }

  get stickyTopPositionFilterText(): string {
    return this.stickyTopPositionFilterNumber + "px";
  }

  get theadHeight(): number {
    return this.thead?.nativeElement ? this.thead?.nativeElement.offsetHeight : 0;
  }

  get tbodyHeight(): number {
    return this.tbody?.nativeElement ? this.tbody?.nativeElement.offsetHeight : 0;
  }

  @Input()
  preloadRowsForPage: boolean = false;

  @Input()
  isMultiSelectable: boolean = false;

  @Input()
  skipInitialQuery: boolean = false;

  @HostBinding("attr.data-test-info-is-loading")
  @Input()
  isLoading: boolean = false;

  @Input()
  displaySpinnerWhenLoading: boolean = false;

  private _queryParameters: QueryParameters;

  @Input()
  columnsSkippedToClear: string[];

  @Input()
  set queryParameters(value: QueryParameters) {
    if (isTrue(this.pendingRetreiveAllForSelection) || isTrue(this.displaySelectAllElementSuccess)) {
      return;
    }
    const newSearchItems = QueryParametersUtil.getSearchItems(value);
    const oldSearchItems = QueryParametersUtil.getSearchItems(this._queryParameters);
    if (this._isDifferentSearchItems(oldSearchItems, newSearchItems)) {
      this.multiSelectionValueOnAllPages = [];
      this.resetCounterItemSelectedOnCurrentPage();
      this.resetBannerSelectAll();
    }
    this._queryParameters = QueryParametersUtil.clone(value);
    if (isNullOrUndefined(value?.paging)) {
      this._queryParameters.paging = new Paging(environment?.defaultPageSize);
    }
    this._computeDisplayButtonCleanAllFilter();
  }

  displayButtonCleanAllFilter: boolean;

  buttonCleanAllFilter: ExtraButtonToolbar<TResource> = {
    color: "primary",
    icon: IconNameEnum.clear,
    displayCondition: current => this.displayButtonCleanAllFilter,
    callback: () => this.clearFilters(),
    labelToTranslate: (resource) => MARK_AS_TRANSLATABLE("table.cleanAllFilter"),
    order: 40,
  };

  private _isDifferentSearchItems(oldSearchItems: MappingObject, newSearchItems: MappingObject): boolean {
    return true;
  }

  private _computeDisplayButtonCleanAllFilter(): void {
    this.displayButtonCleanAllFilter = this.isFilterApplied();
  }

  get queryParameters(): QueryParameters {
    return this._queryParameters;
  }

  private _columns: DataTableColumns[];

  @Input()
  set columns(value: DataTableColumns[]) {
    this._columns = value;
    const defaultSort = this._columns.find(c => isTrue(c.isSortable) && c.order !== 0);
    if (isNotNullNorUndefined(defaultSort)) {
      this.currentSort = {
        field: this.getSortableField(defaultSort),
        order: defaultSort.order,
      } as Sort;
    }
  }

  get columns(): DataTableColumns[] {
    return this._columns;
  }

  private _actions: DataTableActions[] = [];

  @Input()
  set actions(value: DataTableActions[]) {
    this._actions = value;
  }

  get actions(): DataTableActions[] {
    return this._actions.sort((a, b) => a.order - b.order);
  }

  get actionsWrapped(): DataTableActions[] {
    return this.actions.filter(a => isNullOrUndefined(a.isWrapped) || isTrue(a.isWrapped)).sort((a, b) => a.order - b.order);
  }

  get actionsUnwrapped(): DataTableActions[] {
    return this.actions.filter(a => isFalse(a.isWrapped)).sort((a, b) => a.order - b.order);
  }

  private _bulkActions: DataTableBulkActions<TResource>[] = [];

  @Input()
  set bulkActions(value: DataTableBulkActions<TResource>[]) {
    this._bulkActions = value;
  }

  get bulkActions(): DataTableBulkActions<TResource>[] {
    return this._bulkActions;
  }

  private _listHighlightingId: string[] = [];

  @Input()
  set listHighlightingId(value: string[]) {
    this._listHighlightingId = isNullOrUndefined(value) ? [] : value;
  }

  get listHighlightingId(): string[] {
    return isNullOrUndefined(this._listHighlightingId) ? [] : this._listHighlightingId;
  }

  @Input()
  isHighlightCondition: undefined | ((current: TResource) => boolean);

  @Input()
  scrollHeight: string | undefined = undefined;

  private _listCdkDropListId: string[];
  listCdkDropListIdPrefixed: string[];

  @Input()
  set listCdkDropListId(value: string[]) {
    this._listCdkDropListId = value;
    this.listCdkDropListIdPrefixed = value.map(v => environment.cdkDropListIdPrefix + v);
  }

  get listCdkDropListId(): string[] {
    return this._listCdkDropListId;
  }

  @Input()
  isDraggable: boolean;

  @Input()
  draggablePreviewMatIcon: IconNameEnum | undefined = undefined;

  @Input()
  draggablePreviewColumn: keyof TResource | string | undefined = undefined;

  @Input()
  alwaysDisplayButtonActionsWrapped: boolean = false;

  private readonly _selectBS: BehaviorSubject<TResource | undefined> = new BehaviorSubject<TResource | undefined>(undefined);
  @Output("selectChange")
  readonly selectObs: Observable<TResource | undefined> = ObservableUtil.asObservable(this._selectBS);

  private readonly _multiSelectionBS: BehaviorSubject<TResource[]> = new BehaviorSubject<TResource[]>([]);
  @Output("multiSelectionChange")
  readonly multiSelectionObs: Observable<TResource[]> = ObservableUtil.asObservable(this._multiSelectionBS);

  private readonly _dragStartBS: BehaviorSubject<TResource | undefined> = new BehaviorSubject<TResource | undefined>(undefined);
  @Output("dragStartChange")
  readonly dragStartObs: Observable<TResource | undefined> = ObservableUtil.asObservable(this._dragStartBS);

  private readonly _dragEndBS: BehaviorSubject<TResource | undefined> = new BehaviorSubject<TResource | undefined>(undefined);
  @Output("dragEndChange")
  readonly dragEndObs: Observable<TResource | undefined> = ObservableUtil.asObservable(this._dragEndBS);

  set multiSelectionValueOnAllPages(value: TResource[]) {
    this._multiSelectionBS.next(isNullOrUndefined(value) ? [] : value);
  }

  get multiSelectionValueOnAllPages(): TResource[] {
    return isNullOrUndefined(this._multiSelectionBS.value) ? [] : this._multiSelectionBS.value;
  }

  multiSelectionValueOnCurrentPageCounter: number = 0;

  private readonly _queryParametersBS: BehaviorSubject<QueryParameters | undefined> = new BehaviorSubject<QueryParameters | undefined>(undefined);
  @Output("queryParametersChange")
  readonly queryParametersObs: Observable<QueryParameters | undefined> = ObservableUtil.asObservable(this._queryParametersBS);

  private readonly _queryParametersWithoutRefreshBS: BehaviorSubject<QueryParameters | undefined> = new BehaviorSubject<QueryParameters | undefined>(undefined);
  @Output("queryParametersWithoutRefreshChange")
  readonly queryParametersWithoutRefreshObs: Observable<QueryParameters | undefined> = ObservableUtil.asObservable(this._queryParametersWithoutRefreshBS);

  @ViewChild("paginator")
  paginator: SharedPaginatorPresentational;

  isDatasPresent: boolean;
  displayPropositionSelectAllResult: boolean = false;
  pendingRetreiveAllForSelection: boolean = false;
  displaySelectAllElementSuccess: boolean = false;

  get fieldTypeEnum(): typeof FieldTypeEnum {
    return FieldTypeEnum;
  }

  get enumUtil(): typeof EnumUtil {
    return EnumUtil;
  }

  get dataTableComponentEnum(): typeof DataTableComponentEnum {
    return DataTableComponentEnum;
  }

  private readonly _SEPARATOR: string = ".";

  constructor(readonly _breakpointService: BreakpointService,
              private readonly _dialog: MatDialog,
              private readonly _changeDetector: ChangeDetectorRef,
              private readonly _notificationService: NotificationService) {
    super();
  }

  ngOnChanges(changes: SimpleChanges): void {
    if (changes.queryParameters || changes.actions || changes.datas) {
      this.computeContext();
    }
    if (changes.queryParameters || changes.datas) {
      this.computeDatasForPage();
    }
  }

  private computeContext(): void {
    this.context = new Context();
    this.context.mapAtLeastOneActionToDisplay = new Map<DataTableActions, boolean>();
    if (isNullOrUndefined(this._datas) || isNullOrUndefined(this._actions)) {
      return;
    }
    this.actions.forEach(action => {
      this.context.mapAtLeastOneActionToDisplay.set(action, this._shouldDisplayActionColumns(action));
    });
  }

  private _shouldDisplayActionColumns(action: DataTableActions): boolean {
    if (isNullOrUndefined(action.displayOnCondition)) {
      return true;
    }
    return this._datas.some(data => isTrue(action.displayOnCondition(data)));
  }

  private computeDatasForPage(): void {
    const count = this.queryParameters && this.queryParameters.paging ? this.queryParameters.paging.pageSize : 0;
    const defaultValue = this._getEmptyData();
    this._datasForPage = new Array(count || 0).fill(defaultValue);
    (this._datas || []).forEach((value, index) => this._datasForPage[index] = value);
  }

  private resetCounterItemSelectedOnCurrentPage(): void {
    this.multiSelectionValueOnCurrentPageCounter = 0;
  }

  private resetBannerSelectAll(): void {
    this.displayPropositionSelectAllResult = false;
    this.displaySelectAllElementSuccess = false;
  }

  private computeNumberItemSelectedOnCurrentPage(): void {
    this.resetCounterItemSelectedOnCurrentPage();
    this.multiSelectionValueOnAllPages.forEach(itemSelected => {
      if (!isNullOrUndefined(this.datas) && !isNullOrUndefined(this.datas.find(data => data.resId === itemSelected.resId))) {
        this.multiSelectionValueOnCurrentPageCounter++;
      }
    });
  }

  private computeIsDatasPresent(): void {
    let isDatasPresent = false;
    if (!isNullOrUndefined(this.datas) && this.datas.length > 0) {
      isDatasPresent = true;
      this.numberLine = this.datas.length;
    } else {
      this.numberLine = 0;
    }
    this.isDatasPresent = isDatasPresent;
  }

  showDetail(row: TResource): void {
    this._selectBS.next(row);
  }

  getCellData(row: TResource, column: DataTableColumns): any {
    const data = this.getData(row, column);
    if (column.translate && column.filterEnum) {
      const dataStr = data + "";
      const enumValue = column.filterEnum.find(f => f.key === dataStr);
      if (enumValue) {
        return enumValue.value;
      }
      return data;
    }
    return this.applyTypeToData(data, column.type);
  }

  getData(row: TResource, column: DataTableColumns): any {
    const field = column.field;
    if (field.includes(this._SEPARATOR)) {
      return ObjectUtil.getNestedValue(row as any, column.field);
    }
    return row[field];
  }

  getResId(row: TResource): string {
    return (row as TResource).resId;
  }

  private applyTypeToData(data: any, type: FieldTypeEnum): any {
    if (isNullOrUndefined(data)) {
      return StringUtil.stringEmpty;
    }
    if (type === FieldTypeEnum.date) {
      return DateUtil.convertDateToDateString(data);
    }
    if (type === FieldTypeEnum.datetime) {
      return DateUtil.convertDateToDateTimeString(data);
    }
    return data;
  }

  backendSort(columns: DataTableColumns<TResource>): void {
    if (isTrue(this.skipInitialQuery)) {
      this.skipInitialQuery = false;
      return;
    }
    if (isNotNullNorUndefined(columns) && !columns.isSortable) {
      return;
    }
    const queryParameters = ObjectUtil.clone(this.queryParameters);
    if (isNotNullNorUndefined(columns)) {
      this.currentSort = this.generateNewSortObject(columns);
    }
    if (isNotNullNorUndefined(this.currentSort)) {
      queryParameters.sort = this.currentSort;
    } else {
      queryParameters.sort = undefined;
    }
    this._queryParametersBS.next(queryParameters);
  }

  getSortIcon(columns: DataTableColumns<TResource>): IconNameEnum {
    if (isNullOrUndefined(this.currentSort)) {
      return IconNameEnum.sortUndefined;
    }
    if (this.getSortableField(columns) !== this.currentSort.field) {
      return IconNameEnum.sortUndefined;
    }
    return this.currentSort.order === OrderEnum.ascending ? IconNameEnum.sortAscending : IconNameEnum.sortDescending;
  }

  private generateNewSortObject(column: DataTableColumns<TResource>): Sort {
    const newSortableField = this.getSortableField(column);
    let newOrder = OrderEnum.ascending;
    if (this.currentSort?.field === newSortableField) {
      newOrder = this.currentSort.order === OrderEnum.ascending ? OrderEnum.descending : OrderEnum.ascending;
    }
    return {
      field: newSortableField,
      order: newOrder,
    } as Sort;
  }

  pageChange($event: Paging): void {
    this.resetCounterItemSelectedOnCurrentPage();
    this.resetBannerSelectAll();
    this.queryParameters.paging = $event;
    this._queryParametersBS.next(this.queryParameters);
  }

  searchChange(col: DataTableColumns, value: string | BaseResource): void {
    if (this.multiSelectionValueOnAllPages.length > 0) {
      this.subscribe(this._dialog.open(SharedDataTableLooseSelectionDialog, {
        data: {
          numberItemSelected: this.multiSelectionValueOnAllPages.length,
        } as SharedDataTableLooseSelectionDialogData,
      }).afterClosed().pipe(
        tap((isConfirmed: boolean | undefined) => {
          if (isNullOrUndefined(isConfirmed)) {
            this._rollbackChangeOnFilter(col);
            return;
          }
          if (isTrue(isConfirmed)) {
            this._searchChange(col, value);
          }
        }),
      ));
    } else {
      this._searchChange(col, value);
    }
  }

  private _searchChange(col: DataTableColumns, value: string | BaseResource): void {
    this.multiSelectionValueOnAllPages = [];
    this.resetCounterItemSelectedOnCurrentPage();
    this.resetBannerSelectAll();

    const fieldName = this.getFilterableField(col);
    if (isEmptyString(value) || isNullOrUndefined(value)) {
      MappingObjectUtil.delete(QueryParametersUtil.getSearchItems(this.queryParameters), fieldName);
    } else {
      if (col.type === FieldTypeEnum.searchableSingleSelect) {
        value = (value as BaseResource).resId;
      }
      MappingObjectUtil.set(QueryParametersUtil.getSearchItems(this.queryParameters), fieldName, value);
    }
    const queryParameters = QueryParametersUtil.resetToFirstPage(this.queryParameters);
    this._queryParametersBS.next(queryParameters);
  }

  private _rollbackChangeOnFilter(col: DataTableColumns<TResource>): void {
    col.isFilterable = false;
    this._changeDetector.detectChanges();
    setTimeout(() => {
      col.isFilterable = true;
      this._changeDetector.detectChanges();
    }, 0);
  }

  getValue(col: DataTableColumns<TResource>): string | null {
    const value = MappingObjectUtil.get(this.queryParameters.search.searchItems, this.getFilterableField(col));
    return isNonEmptyString(value) ? value : null;
  }

  private getFilterableField(col: DataTableColumns<TResource>): string {
    return isString(col.filterableField) ? col.filterableField : col.field;
  }

  private getSortableField(col: DataTableColumns): string {
    return isString(col.sortableField) ? col.sortableField : col.field;
  }

  getSortableFieldByFieldName(fieldName: string): string {
    const column = this.columns.find(col => col.field === fieldName);
    return this.getSortableField(column);
  }

  isFilterApplied(): boolean {
    let toSkip: boolean = false;
    if (MappingObjectUtil.size(QueryParametersUtil.getSearchItems(this.queryParameters)) > 0) {
      if (this.columnsSkippedToClear === undefined || isEmptyArray(this.columnsSkippedToClear)) {
        return true;
      }
      const searchItems: Map<string, string> = MappingObjectUtil.toMap(this.queryParameters.search.searchItems);
      searchItems.forEach((value: string, key: string) => {
        if (!toSkip && this.columnsSkippedToClear.some(c => c.includes(key)) === false) {
          toSkip = true;
        }
      });
    }
    return toSkip;
  }

  clearFilters(): void {
    if (this.multiSelectionValueOnAllPages.length > 0) {
      this.subscribe(this._dialog.open(SharedDataTableLooseSelectionDialog, {
        data: {
          numberItemSelected: this.multiSelectionValueOnAllPages.length,
        } as SharedDataTableLooseSelectionDialogData,
      }).afterClosed().pipe(
        tap((isConfirmed: boolean | undefined) => {
          if (isNullOrUndefined(isConfirmed)) {
            return;
          }
          if (isTrue(isConfirmed)) {
            this._clearFilters();
          }
        }),
      ));
    } else {
      this._clearFilters();
    }
  }

  private _clearFilters(): void {
    this.multiSelectionValueOnAllPages = [];
    this.resetCounterItemSelectedOnCurrentPage();
    this.resetBannerSelectAll();

    let queryParameters = QueryParametersUtil.clone(this.queryParameters);
    const searchItems = QueryParametersUtil.getSearchItems(queryParameters);
    if (this.columnsSkippedToClear === undefined || isEmptyArray(this.columnsSkippedToClear)) {
      MappingObjectUtil.clear(searchItems);
    } else {
      MappingObjectUtil.forEach(searchItems, (value: string, key: string) => {
        if (this.columnsSkippedToClear.some(c => c.includes(key)) === false) {
          MappingObjectUtil.delete(searchItems, key);
        }
      });
    }
    queryParameters = QueryParametersUtil.resetToFirstPage(queryParameters);
    this._queryParametersBS.next(queryParameters);
  }

  callCallback($event: MouseEvent | KeyboardEvent, callback: (current: TResource) => void, rowData: TResource): void {
    this.avoidOpen($event);
    callback(rowData);
  }

  isHighlighting(rowData: TResource): boolean {
    return this.listHighlightingId.includes(rowData.resId) || this.multiSelectionValueOnAllPages.findIndex(s => s.resId === rowData.resId) !== -1
      || (isNotNullOrUndefined(this.isHighlightCondition) && this.isHighlightCondition(rowData));
  }

  displayActionOnCondition(action: DataTableActions<TResource>, resource: TResource = undefined): boolean {
    if (!resource || !action.displayOnCondition) {
      return this.atLeastOneActionRowDisplayed(action);
    }
    return action.displayOnCondition(resource);
  }

  displayActionsWrapperOnCondition(actions: DataTableActions<TResource>[], resource: TResource): boolean {
    return actions.some(action => isFunction(action.displayOnCondition) && action.displayOnCondition(resource));
  }

  private atLeastOneActionRowDisplayed(action: DataTableActions<TResource>): boolean {
    if (!isNonEmptyArray(this._datas)) {
      return false;
    }
    if (isNullOrUndefined(action.displayOnCondition)) {
      return true;
    }
    return this.context.mapAtLeastOneActionToDisplay.get(action);
  }

  getWidth(column: DataTableColumns<TResource>): string {
    if (isString(column.width)) {
      return column.width;
    }
    if (column.component === DataTableComponentEnum.conformityLevelStar) {
      return "100px";
    }
    if (column.type === FieldTypeEnum.datetime) {
      return "175px";
    }
    if (column.type === FieldTypeEnum.date) {
      return "155px";
    }

    return undefined;
  }

  getMinWidth(column: DataTableColumns<TResource>): string {
    if (isString(column.minWidth)) {
      return column.minWidth;
    }
    return undefined;
  }

  getMaxWidth(column: DataTableColumns<TResource>): string {
    if (isString(column.maxWidth)) {
      return column.maxWidth;
    }
    return undefined;
  }

  isColumnFilterable(): boolean {
    return this.columns.some(c => c.isFilterable);
  }

  toggleCheckbox($event: MatCheckboxChange, rowData: TResource): void {
    const listRowSelected = this._multiSelectionBS.value;
    if ($event.checked) {
      listRowSelected.push(rowData);
      this.multiSelectionValueOnCurrentPageCounter++;
    } else {
      const index = listRowSelected.findIndex(r => r.resId === rowData.resId);
      if (index === -1) {
        return;
      }
      this.multiSelectionValueOnCurrentPageCounter--;
      listRowSelected.splice(index, 1);
    }
    this._multiSelectionBS.next(listRowSelected);
  }

  avoidOpen($event: MouseEvent | KeyboardEvent): void {
    $event.stopPropagation();
  }

  toggleCurrentPageCheckbox($event: MatCheckboxChange): void {
    const listRowSelectedOnAllPages = this.multiSelectionValueOnAllPages;
    this.datas.forEach((item: TResource) => {
      const indexOf = listRowSelectedOnAllPages.findIndex(r => r.resId === item.resId);
      if ($event.checked) {
        this.displayPropositionSelectAllResult = true;
        if (indexOf !== -1) {
          return;
        }
        listRowSelectedOnAllPages.push(item);
      } else {
        this.resetBannerSelectAll();
        if (indexOf === -1) {
          return;
        }
        listRowSelectedOnAllPages.splice(indexOf, 1);
      }
    });
    this._multiSelectionBS.next(listRowSelectedOnAllPages);
    this.computeNumberItemSelectedOnCurrentPage();
    if ($event.checked && this.datas.length === this.queryParameters.paging.length) {
      this.resetBannerSelectAll();
    }
  }

  isLineSelected(rowData: TResource): boolean {
    return this.multiSelectionValueOnAllPages.findIndex(m => m.resId === rowData.resId) !== -1;
  }

  selectAllResult(): Observable<string[]> {
    if (this.queryParameters.paging.length > environment.maximalPageSizeToRetrievePaginationInfo) {
      this._notificationService.showInformation(MARK_AS_TRANSLATABLE("table.notification.onlyTheMaximalAmountAreSelected"), {maximalNumber: environment.maximalPageSizeToRetrievePaginationInfo} as any);
    }
    this.pendingRetreiveAllForSelection = true;
    const queryParameters = QueryParametersUtil.clone(this.queryParameters);
    queryParameters.paging = new Paging(environment.maximalPageSizeToRetrievePaginationInfo);
    this._queryParametersBS.next(queryParameters);
    this._queryParametersWithoutRefreshBS.next(QueryParametersUtil.clone(this.queryParameters));
    return this.multiSelectionObs.pipe(
      map(list => list.map(m => m.resId)),
    );
  }

  cleanSelection(): void {
    this.resetBannerSelectAll();
    this.multiSelectionValueOnAllPages = [];
    this.resetCounterItemSelectedOnCurrentPage();
    this._changeDetector.detectChanges();
  }

  callBulkAction(bulkAction: DataTableBulkActions<TResource>): void {
    const result = bulkAction.callback(this.multiSelectionValueOnAllPages);
    if (isObservable(result)) {
      this.subscribe(result.pipe(take(1)), confirmed => {
        if (isTrue(confirmed)) {
          this.cleanSelection();
        }
      });
    } else {
      this.cleanSelection();
    }
  }

  dragStart($event: CdkDragStart<TResource>): void {
    this.cleanSelection();
    this._dragStartBS.next($event.source.data);
  }

  dragEnd($event: CdkDragEnd<TResource>): void {
    this._dragEndBS.next($event.source.data);
  }

  cleanFilter(select: MatSelect | HTMLInputElement, col: DataTableColumns<TResource>): void {
    select.value = StringUtil.stringEmpty;
    this.searchChange(col, StringUtil.stringEmpty);
  }

  isDisabledAction(action: DataTableActions<TResource>, rowData: TResource): ObservableOrPromiseOrValue<boolean> {
    if (isNullOrUndefined(action.disableCondition)) {
      return false;
    }
    return action.disableCondition(rowData);
  }

  isVisible(action: DataTableActions<TResource>): ObservableOrPromiseOrValue<boolean> {
    if (isNullOrUndefined(action.isVisible)) {
      return false;
    }
    return action.isVisible;
  }

  calculateClasses(action: DataTableActions<TResource>): any {
    return {
      "action-button": true,
      "visible": this.isVisible(action),
    };
  }

  // Empty data

  private static readonly _emptyData: {} = {};

  @Input()
  focusFirstElement: boolean = true;

  static isEmptyData(value: any): boolean {
    return this._emptyData === value;
  }

  private _getEmptyData(): {} {
    return SharedDataTablePresentational._emptyData;
  }

  isEmptyData(value: any): boolean {
    return SharedDataTablePresentational.isEmptyData(value);
  }

  getTooltip(col: DataTableColumns, value: string): string {
    if (isNullOrUndefined(col.tooltip)) {
      return value;
    }
    return col.tooltip(value);
  }

  refresh(): void {
    this._queryParametersBS.next(this.queryParameters);
  }

  createFormControl(key: string): FormControl {
    return new FormControl(key);
  }

  preventDefault($event: MouseEvent): void {
    $event.preventDefault();
    $event.stopPropagation();
  }

  setForceHoverRowData(rowData: TResource | undefined): void {
    this.forceHoverRowData = rowData;
  }
}

class Context {
  mapAtLeastOneActionToDisplay: Map<DataTableActions, boolean>;
}
