import {
  ChangeDetectionStrategy,
  ChangeDetectorRef,
  Component,
  Input,
  OnInit,
} from "@angular/core";
import {Enums} from "@enums";
import {TranslateService} from "@ngx-translate/core";
import {Store} from "@ngxs/store";
import {SharedAbstractPresentational} from "@shared/components/presentationals/shared-abstract/shared-abstract.presentational";
import {IconNameEnum} from "@shared/enums/icon-name.enum";
import {LabelTranslateEnum} from "@shared/enums/label-translate.enum";
import {
  DataSensitivityColor,
  DataSensitivityHelper,
} from "@shared/helpers/data-sensitivity.helper";
import {KeyValueInfo} from "@shared/models/key-value-info.model";
import {
  distinctUntilChanged,
  tap,
} from "rxjs/operators";
import {
  EnumUtil,
  isNotNullNorUndefined,
  isNullOrUndefined,
  StringUtil,
} from "solidify-frontend";

@Component({
  selector: "dlcm-shared-data-sensitivity",
  templateUrl: "./shared-data-sensitivity.presentational.html",
  styleUrls: ["./shared-data-sensitivity.presentational.scss"],
  changeDetection: ChangeDetectionStrategy.OnPush,
})
export class SharedDataSensitivityPresentational extends SharedAbstractPresentational implements OnInit {
  _dataSensitivity: Enums.Deposit.DataSensitivityEnum;
  dataSensitivityIconName: IconNameEnum = IconNameEnum.dataSensitivity;
  dataSensitivityColor: DataSensitivityColor;
  label: string;

  icon: IconNameEnum = this.dataSensitivityIconName;

  @Input()
  set dataSensitivity(dataSensitivity: Enums.Deposit.DataSensitivityEnum) {
    this._dataSensitivity = dataSensitivity;
    this._computeDataSensitivityColor(dataSensitivity);
    this._computeLabel(dataSensitivity);
    this._computeIcon(dataSensitivity);
  }

  get dataSensitivity(): Enums.Deposit.DataSensitivityEnum {
    return this._dataSensitivity;
  }

  get dataSensitivityHelper(): typeof DataSensitivityHelper {
    return DataSensitivityHelper;
  }

  @Input()
  size: string = "xlg";

  @Input()
  withTooltip: boolean = false;

  constructor(private readonly _changeDetector: ChangeDetectorRef,
              private readonly _translateService: TranslateService,
              private readonly _store: Store,
              private readonly _cd: ChangeDetectorRef) {
    super();
  }

  ngOnInit(): void {
    super.ngOnInit();
    this.subscribe(this._translateService.onLangChange.asObservable().pipe(
      distinctUntilChanged(),
      tap(language => {
        this._computeLabel(this.dataSensitivity);
        this._cd.detectChanges();
      }),
    ));
  }

  private _computeDataSensitivityColor(dataSensitivity: Enums.Deposit.DataSensitivityEnum): void {
    this.dataSensitivityColor = DataSensitivityHelper.getColor(dataSensitivity);
  }

  private _computeLabel(dataSensitivity: Enums.Deposit.DataSensitivityEnum): void {
    if (isNullOrUndefined(dataSensitivity)) {
      this.label = StringUtil.stringEmpty;
      return;
    }
    let partiallySupported = "";
    const keyValueInfos = EnumUtil.getKeyValue(Enums.Deposit.DataSensitivityEnumTranslate, dataSensitivity) as KeyValueInfo;
    if (DataSensitivityHelper.isPartiallySupported(dataSensitivity)) {
      partiallySupported = " (" + this._translateService.instant(LabelTranslateEnum.partiallySupported) + ")";
    }
    this.label = this._translateService.instant(keyValueInfos.value) + partiallySupported +
      (isNotNullNorUndefined(keyValueInfos.infoToTranslate) ? " : " + this._translateService.instant(keyValueInfos.infoToTranslate) : StringUtil.stringEmpty);
  }

  private _computeIcon(dataSensitivity: Enums.Deposit.DataSensitivityEnum): void {
    if (DataSensitivityHelper.isPartiallySupported(dataSensitivity)) {
      switch (dataSensitivity) {
        case  Enums.Deposit.DataSensitivityEnum.ORANGE:
          this.icon = IconNameEnum.dataSensitivityPartiallySupportedOrange;
          break;
        case  Enums.Deposit.DataSensitivityEnum.RED:
          this.icon = IconNameEnum.dataSensitivityPartiallySupportedRed;
          break;
        case  Enums.Deposit.DataSensitivityEnum.CRIMSON:
          this.icon = IconNameEnum.dataSensitivityPartiallySupportedCrimson;
          break;
      }
    } else {
      this.icon = this.dataSensitivityIconName;
    }
  }
}
