import {
  ChangeDetectionStrategy,
  Component,
  Input,
  OnInit,
} from "@angular/core";
import {
  ControlValueAccessor,
  FormControl,
  NG_VALUE_ACCESSOR,
} from "@angular/forms";
import {Enums} from "@enums";
import {SharedAbstractPresentational} from "@shared/components/presentationals/shared-abstract/shared-abstract.presentational";
import {LabelTranslateEnum} from "@shared/enums/label-translate.enum";
import {
  distinctUntilChanged,
  tap,
} from "rxjs/operators";
import {
  EnumUtil,
  isNullOrUndefined,
} from "solidify-frontend";

@Component({
  selector: "dlcm-shared-compliance-level-rating",
  templateUrl: "./shared-compliance-level.presentational.html",
  styleUrls: ["./shared-compliance-level.presentational.scss"],
  changeDetection: ChangeDetectionStrategy.OnPush,
  providers: [
    {
      provide: NG_VALUE_ACCESSOR,
      multi: true,
      useExisting: SharedComplianceLevelPresentational,
    },
  ],
})

export class SharedComplianceLevelPresentational extends SharedAbstractPresentational implements OnInit, ControlValueAccessor {
  @Input()
  formControl: FormControl;

  @Input()
  set value(value: Enums.DataFile.ComplianceLevelEnum) {
    this._value = value;
    this.computeComplianceLevelInfos();
  }

  get value(): Enums.DataFile.ComplianceLevelEnum {
    return this._value;
  }

  _value: Enums.DataFile.ComplianceLevelEnum;

  @Input()
  withLabel: boolean = true;

  @Input()
  center: boolean = false;

  rating: number = 0;

  maxStarsNumber: number = 3;
  label: string | undefined = LabelTranslateEnum.complianceLevel;
  tooltipToTranslate: string | undefined;

  hasBeenAssessed: boolean = true;

  ngOnInit(): void {
    if (!isNullOrUndefined(this.formControl)) {
      this.subscribe(this.formControl.valueChanges.pipe(
        distinctUntilChanged(),
        tap(value => {
          this.value = value;
        }),
      ));
      this.value = this.formControl.value;
    }
  }

  computeComplianceLevelInfos(): void {
    const complianceLevel: Enums.DataFile.ComplianceLevelEnum = this.value;
    this.tooltipToTranslate = EnumUtil.getLabel(Enums.DataFile.ComplianceLevelEnumTranslate, complianceLevel);
    const complianceLevelNumber = Enums.DataFile.ConvertComplianceLevel.convertStringToNumber(complianceLevel);
    this.rating = complianceLevelNumber < 0 ? complianceLevelNumber : complianceLevelNumber - 1;
    this.computeHasBeenAssessed();
  }

  computeHasBeenAssessed(): void {
    this.hasBeenAssessed = this.value === Enums.DataFile.ComplianceLevelEnum.NOT_ASSESSED;
  }

  writeValue(value: Enums.DataFile.ComplianceLevelEnum): void {
  }

  registerOnChange(fn: any): void {
  }

  registerOnTouched(fn: any): void {
  }

  setDisabledState?(isDisabled: boolean): void {
  }
}
